<?php

namespace App\Exports;

use App\Models\Product;
use Illuminate\Database\Eloquent\Collection;
use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithStyles;
use PhpOffice\PhpSpreadsheet\Worksheet\Worksheet;

class ProductsExport implements FromCollection, WithHeadings, WithStyles
{
    protected $type;
    protected $category;
    protected $brand;

    public function __construct($type, $category, $brand)
    {
        $this->type = $type;
        $this->category = $category;
        $this->brand = $brand;
    }

    /**
     * @return \Illuminate\Support\Collection
     */
    public function collection()
    {
        $type = $this->type;
        $category = $this->category;
        $brand = $this->brand;

        $products = Product::query()->where('status', '!=', 'deleted')->where('status', '!=', 'deep_deleted');

        // inventory type filter
        if ($type == 'instock') {
            $products = $products->whereHas('inventories', function ($query) {
                $query->where('status', 'publish')->where('count', '>', 0);
            });
        } else if ($type == 'outofstock') {
            $products = $products->where(function ($q) {
                $q->whereHas('inventories', function ($query) {
                    $query->where('status', 'publish')->where('count', 0);
                })->orWhereDoesntHave('inventories');
            });
        } else if ($type == 'vip') {
            $products = $products->where('is_vip', true);
        } else if ($type == 'publish') {
            $products = $products->where('status', 'publish');
        } else if ($type == 'draft') {
            $products = $products->where('status', 'draft');
        }

        // category filter
        if ($category != 'all' && $category != null && $category != '') {
            $products = $products->where('category_id', $category);
        }

        // brand filter
        if ($brand != 'all' && $brand != null && $brand != '') {
            $products = $products->where('brand_id', $brand);
        }

        // order by desc
        $products = $products->orderBy('created_at', 'desc')->get();

        // add inventories to final list
        $result = new Collection();
        foreach ($products as $product) {
            $result->add([
                'action_type' => 'update',
                'uuid' => $product->uuid,
                'title' => $product->title,
                'title_en' => $product->title_en,
                'slug' => $product->slug,
                'category' => $product->productCategory->title,
                'brand' => $product->brand?->title,
                'warnings' => $product->warnings,
                'content' => $product->content,
                'guide' => $product->guide,
                'images' => is_array($product->get_all_images) ? json_encode($product->get_all_images, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES) : null,
                'attrs' => is_array($product->get_attrs) ? json_encode($product->get_attrs, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES) : null,
                'seo_title' => $product->seo_title,
                'seo_description' => $product->seo_description,
                'view_count' => $product->view_count,
                'commission' => $product->commission > 0 ? $product->commission : '0',
                'is_vip' => $product->is_vip ? '1' : '0',
                'price_model' => is_array($product->raw_price_model) ? json_encode($product->raw_price_model, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES) : null,
                'status' => $product->status,
                'last_update' => $product->jalali_updated_at[2],
            ]);
        }

        return $result;
    }

    public function headings(): array
    {
        return [
            'نوع عملیات',
            'شناسه محصول',
            'نام محصول',
            'نام انگلیسی',
            'نامک',
            'دسته‌بندی',
            'برند',
            'هشدار ها',
            'محتوا',
            'راهنمای ویژگی',
            'تصاویر',
            'ویژگی‌های برجسته',
            'عنوان سئو',
            'توضیح سئو',
            'میزان بازدید',
            'کمیسیون',
            'ویژه',
            'متغیر‌های فروش',
            'وضعیت',
            'آخرین ویرایش',
        ];
    }

    public function styles(Worksheet $sheet)
    {
        $sheet->setRightToLeft(true);

        $highestColumn = $sheet->getHighestColumn();
        $highestColumnIndex = \PhpOffice\PhpSpreadsheet\Cell\Coordinate::columnIndexFromString($highestColumn);

        for ($col = 1; $col <= $highestColumnIndex; $col++) {
            $columnLetter = \PhpOffice\PhpSpreadsheet\Cell\Coordinate::stringFromColumnIndex($col);
            if ($columnLetter == 'B' || $columnLetter == 'C') {
                $sheet->getColumnDimension($columnLetter)->setAutoSize(true);
            }
        }

        $highestRow = $sheet->getHighestRow();
        $sheet->getStyle("A1:{$highestColumn}{$highestRow}")->getFont()->setName('Tahoma')->setSize(11);
    }
}
