<?php

namespace App\Http\Controllers\Admin\Blog;

use App\Models\ArticleCategory;
use App\Http\Controllers\Controller;
use App\Http\Requests\Admin\ArticleCategoryRequest;
use Illuminate\Http\Request;
use Inertia\Inertia;

class ArticleCategoriesController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $categories = ArticleCategory::where('status', 'publish')->withCount('articles')->get()->map(fn($category) => [
            'id' => $category->id,
            'title' => $category->title,
            'slug' => $category->slug,
            'get_formatted_view_count' => $category->get_formatted_view_count,
            'articles_count' => $category->articles_count,
        ]);

        return Inertia::render('Admin/Blog/Categories/List', [
            'categories' => $categories,
        ]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return Inertia::render('Admin/Blog/Categories/Create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(ArticleCategoryRequest $request)
    {
        $category = ArticleCategory::create([
            'title' => $request->title,
            'slug' => $request->exists('slug') ? str_replace('/', '-', str_replace('\\', '-', str_replace('%', '', $request->slug))) : null,
            'image' => $request->exists('image') ? $request->image : null,
            'seo_title' => $request->exists('seo_title') ? $request->seo_title : null,
            'seo_description' => $request->exists('seo_description') ? $request->seo_description : null,
            'seo_keywords' => $request->exists('seo_keywords') ? $request->seo_keywords : null,
            'seo_canonical' => $request->exists('seo_canonical') ? urldecode($request->seo_canonical) : null,
            'status' => 'publish',
        ]);

        /** Report Action */
        $this->report(__("messages.report_category_created"), 'article_category', $category);

        return redirect()->route('admin.article.categories.index')->with('message', [__('messages.category_created')]);
    }

    /**
     * Display the specified resource.
     */
    public function show(ArticleCategory $articleCategory)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(ArticleCategory $category)
    {
        return Inertia::render('Admin/Blog/Categories/Edit', [
            'category' => $category,
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(ArticleCategoryRequest $request, ArticleCategory $category)
    {
        $category->update([
            'title' => $request->title,
            'slug' => $request->exists('slug') ? str_replace('/', '-', str_replace('\\', '-', str_replace('%', '', $request->slug))) : $category->slug,
            'image' => $request->exists('image') ? $request->image : $category->image,
            'seo_title' => $request->exists('seo_title') ? $request->seo_title : $category->seo_title,
            'seo_description' => $request->exists('seo_description') ? $request->seo_description : $category->seo_description,
            'seo_keywords' => $request->exists('seo_keywords') ? $request->seo_keywords : $category->seo_keywords,
            'seo_canonical' => $request->exists('seo_canonical') ? urldecode($request->seo_canonical) : $category->seo_canonical,
            'status' => 'publish',
        ]);

        /** Report Action */
        $this->report(__("messages.report_category_updated"), 'article_category', $category);

        return redirect()->route('admin.article.categories.edit', $category->slug)->with('message', [__('messages.category_updated')]);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(ArticleCategory $category)
    {
        if ($category->articles()->count() > 0) {
            return redirect()->route('admin.article.categories.index')->withErrors([__('messages.category_has_article')]);
        }

        $category->delete();

        /** Report Action */
        $this->report(__("messages.report_category_deleted"), 'article_category', $category);

        return redirect()->route('admin.article.categories.index')->with('message', [__('messages.category_deleted')]);
    }
}
