<?php

namespace App\Http\Controllers\Admin\Forms;

use App\Helpers\DropdownListHelper;
use App\Helpers\UUIDHelper;
use App\Models\Form;
use App\Http\Controllers\Controller;
use App\Http\Requests\Admin\FormsRequest;
use App\Models\FormAnswer;
use App\Services\Admin\Forms\FormsService;
use Illuminate\Http\Request;
use Inertia\Inertia;

class FormsController extends Controller
{
    use DropdownListHelper, UUIDHelper;

    /**
     * Display a listing of the resource.
     */
    public function index(Request $request, FormsService $formsService)
    {
        // init system forms
        $formsService->createSystemForms();

        return Inertia::render('Admin/Forms/List', [
            'forms' => $formsService->getForms($request),
        ]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return Inertia::render('Admin/Forms/Create', [
            'statusList' => $this->contentStatusList(),
            'formTypeList' => $this->formTypeList(),
            'formPriorityList' => $this->formPriorityList(),
        ]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(FormsRequest $request, FormsService $formsService)
    {
        /** validate fields */
        switch ($formsService->validateFields($request)) {
            case "form_fields_empty":
                return redirect()->back()->withErrors([__('messages.form_fields_empty')]);
                break;
            case "form_field_empty":
                return redirect()->back()->withErrors([__('messages.form_field_empty')]);
                break;
        }

        /** validate duplicated slug */
        if ($request->exists('slug') && $request->slug != null) {
            $formWithSlug = Form::where('slug', $request->slug)->first();
            if ($formWithSlug != null) {
                return redirect()->back()->withErrors([__('messages.slug_is_exist')]);
            }
        }

        $form = Form::create([
            'uuid' => $this->generateUniqueRandomNumber(6, \App\Models\Form::class, 'uuid', 'form-'),
            'title' => $request->title,
            'slug' => $request->exists('slug') && $request->slug != null ? $request->slug : null,
            'description' => $request->exists('description') ? $request->description : null,

            'fields' => serialize($request->fields),

            'seo_title' => $request->exists('seo_title') ? $request->seo_title : null,
            'seo_description' => $request->exists('seo_description') ? $request->seo_description : null,
            'seo_keywords' => $request->exists('seo_keywords') ? $request->seo_keywords : null,
            'seo_canonical' => $request->exists('seo_canonical') ? urldecode($request->seo_canonical) : null,

            'flag' => 'normal',
            'status' => $request->status,
        ]);

        /** Report Action */
        $this->report(__("messages.report_form_created"), 'form', $form);

        return redirect()->route('admin.forms.index')->with('message', [__('messages.form_created')]);
    }

    /**
     * Display the specified resource.
     */
    public function show(Request $request, Form $form, FormsService $formsService)
    {
        // check read
        $form->formAnswers()->where('flag', 'unread')->update(['flag' => 'read']);

        return Inertia::render('Admin/Forms/Answers', [
            'form' => $form,
            'answers' => $formsService->getFormAnswers($form),
        ]);
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Form $form)
    {
        return Inertia::render('Admin/Forms/Edit', [
            'form' => $form,
            'statusList' => $this->contentStatusList(),
            'formTypeList' => $this->formTypeList(),
            'formPriorityList' => $this->formPriorityList(),
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(FormsRequest $request, Form $form, FormsService $formsService)
    {
        /** validate fields */
        switch ($formsService->validateFields($request)) {
            case "form_fields_empty":
                return redirect()->back()->withErrors([__('messages.form_fields_empty')]);
                break;
            case "form_field_empty":
                return redirect()->back()->withErrors([__('messages.form_field_empty')]);
                break;
        }

        /** validate duplicated slug */
        if ($request->exists('slug') && $request->slug != null && $request->slug != $form->slug) {
            $formWithSlug = Form::where('slug', $request->slug)->first();
            if ($formWithSlug != null) {
                return redirect()->back()->withErrors([__('messages.slug_is_exist')]);
            }
        }

        $form->update([
            'title' => $request->title,
            'slug' => $request->exists('slug') && $form->flag != 'system' ? $request->slug : $form->slug,
            'description' => $request->exists('description') ? $request->description : $form->description,

            'fields' => serialize($request->fields),

            'seo_title' => $request->exists('seo_title') ? $request->seo_title : $form->seo_title,
            'seo_description' => $request->exists('seo_description') ? $request->seo_description : $form->seo_description,
            'seo_keywords' => $request->exists('seo_keywords') ? $request->seo_keywords : $form->seo_keywords,
            'seo_canonical' => $request->exists('seo_canonical') ? urldecode($request->seo_canonical) : $form->seo_canonical,

            'status' => $request->status,
        ]);

        /** Report Action */
        $this->report(__("messages.report_form_updated"), 'form', $form);

        return redirect()->back()->with('message', [__('messages.form_updated')]);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Request $request, Form $form)
    {
        if ($form->flag == 'system') {
            return redirect()->back()->withErrors([__('messages.unknown_error')]);
        }

        if ($request->status == 'deleted') {
            $form->delete();

            /** Report Action */
            $this->report(__("messages.report_form_deleted"), 'form', $form);

            return redirect()->back()->with('message', [__('messages.form_deleted')]);
        } else if ($form->status != 'publish' && $request->status == 'publish') {
            $form->update([
                'status' => 'publish'
            ]);

            /** Report Action */
            $this->report(__("messages.report_form_publish"), 'form', $form);

            return redirect()->back()->with('message', [__('messages.form_publish')]);
        } else if ($form->status != 'draft' && $request->status == 'draft') {
            $form->update([
                'status' => 'draft'
            ]);

            /** Report Action */
            $this->report(__("messages.report_form_draft"), 'form', $form);

            return redirect()->back()->with('message', [__('messages.form_draft')]);
        }

        return redirect()->back()->withErrors([__('messages.unknown_error')]);
    }

    /**
     * remove form answer
     */
    public function answer_destroy(Form $form, FormAnswer $answer)
    {
        $answer->delete();

        /** Report Action */
        $this->report(__("messages.report_form_answer_deleted"), 'form', $form);

        return redirect()->back()->with('message', [__('messages.form_answer_deleted')]);
    }
}
