<?php

namespace App\Http\Controllers\Admin\Products;

use App\Helpers\DropdownListHelper;
use App\Helpers\UUIDHelper;
use App\Models\Brand;
use App\Http\Controllers\Controller;
use App\Http\Requests\Admin\BrandsRequest;
use App\Services\Admin\Products\BrandsService;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Inertia\Inertia;
use Illuminate\Validation\Rule;

class BrandsController extends Controller
{
    use UUIDHelper;

    /**
     * Display a listing of the resource.
     */
    public function index(Request $request, BrandsService $brandsService)
    {
        return Inertia::render('Admin/Products/Brand/List', [
            'brands' => $brandsService->getBrands($request)
        ]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return Inertia::render('Admin/Products/Brand/Create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(BrandsRequest $request): RedirectResponse
    {
        /** @var App\Models\User $user */
        $user = auth()->user();

        $brand = $user->brands()->create([
            'uuid' => $this->generateUniqueRandomNumber(6, \App\Models\Brand::class, 'uuid', 'brand-'),
            'title' => $request->title,
            'title_en' => $request->exists('title_en') ? $request->title_en : null,
            'website' => $request->exists('website') ? $request->website : null,
            'slug' => $request->exists('slug') ? $request->slug : null,
            'content' => $request->content,
            'banner' => $request->exists('banner') ? $request->banner : null,
            'logo' => $request->exists('logo') ? $request->logo : null,
            'seo_title' => $request->exists('seo_title') ? $request->seo_title : null,
            'seo_description' => $request->exists('seo_description') ? $request->seo_description : null,
            'seo_keywords' => $request->exists('seo_keywords') ? $request->seo_keywords : null,
            'seo_canonical' => $request->exists('seo_canonical') ? urldecode($request->seo_canonical) : null,
            'status' => 'publish',
        ]);

        /** Report Action */
        $this->report(__("messages.report_brand_created"), 'brand', $brand);

        return redirect()->route('admin.brands.index')->with('message', [__('messages.brand_created')]);
    }

    /**
     * Display the specified resource.
     */
    public function show(Brand $brand)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Brand $brand)
    {
        return Inertia::render('Admin/Products/Brand/Edit', [
            'brand' => $brand,
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(BrandsRequest $request, Brand $brand)
    {
        $request->validate([
            'slug' => ['nullable', Rule::unique(Brand::class)->ignore($brand->id)],
        ]);

        $brand->update([
            'title' => $request->title,
            'title_en' => $request->exists('title_en') ? $request->title_en : $brand->title_en,
            'website' => $request->exists('website') ? $request->website : $brand->website,
            'slug' => $request->has('slug') ? str_replace('/', '-', str_replace('\\', '-', str_replace('%', '', $request->slug))) : $brand->slug,
            'content' => $request->content,
            'banner' => $request->exists('banner') ? $request->banner : $brand->banner,
            'logo' => $request->exists('logo') ? $request->logo : $brand->logo,
            'seo_title' => $request->exists('seo_title') ? $request->seo_title : $brand->seo_title,
            'seo_description' => $request->exists('seo_description') ? $request->seo_description : $brand->seo_description,
            'seo_keywords' => $request->exists('seo_keywords') ? $request->seo_keywords : $brand->seo_keywords,
            'seo_canonical' => $request->exists('seo_canonical') ? urldecode($request->seo_canonical) : $brand->seo_canonical,
        ]);

        /** Report Action */
        $this->report(__("messages.report_brand_updated"), 'brand', $brand);

        return redirect()->route('admin.brands.edit', $brand->slug)->with('message', [__('messages.brand_updated')]);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Request $request, Brand $brand)
    {
        if ($brand->status == "publish" && $brand->products()->count() == 0) {
            $brand->update([
                'status' => 'deleted'
            ]);

            /** Report Action */
            $this->report(__("messages.report_brand_to_trash"), 'brand', $brand);

            return redirect()->route('admin.brands.index')->with('message', [__('messages.brand_to_trash')]);
        } else if ($brand->status == "publish") {
            $brand->update([
                'status' => 'archive'
            ]);

            /** Report Action */
            $this->report(__("messages.report_brand_archived"), 'brand', $brand);

            return redirect()->route('admin.brands.index')->with('message', [__('messages.brand_archived')]);
        } else if ($brand->status == "archive") {
            $brand->update([
                'status' => 'publish'
            ]);

            /** Report Action */
            $this->report(__("messages.report_brand_published"), 'brand', $brand);

            return redirect()->route('admin.brands.index')->with('message', [__('messages.brand_published')]);
        } else if ($request->exists('status') && $request->status == "publish" && $brand->status == "deleted") {
            $brand->update([
                'status' => 'publish'
            ]);

            /** Report Action */
            $this->report(__("messages.report_brand_published"), 'brand', $brand);

            return redirect()->route('admin.brands.trash')->with('message', [__('messages.brand_published')]);
        } else if ($request->exists('status') && $request->status == "delete" && $brand->status == "deleted") {
            $brand->delete();

            /** Report Action */
            $this->report(__("messages.report_brand_deleted"), 'brand', $brand);

            return redirect()->route('admin.brands.trash')->with('message', [__('messages.brand_deleted')]);
        } else {
            return redirect()->route('admin.brands.index')->with('message', [__('messages.unknown_error')]);
        }
    }

    /**
     * show trash
     */
    public function trash(Request $request, BrandsService $brandsService)
    {
        return Inertia::render('Admin/Products/Brand/Trash', [
            'brands' => $brandsService->getTrashBrands($request)
        ]);
    }

    /** search raw brand */
    public function searchBrand(Request $request)
    {
        $brands = Brand::query()->where('status', 'publish')->when($request->input('search'), function ($query, $search) {
            $query->where('title', 'like', "%{$search}%");
        })->orderBy('created_at', 'desc')->pluck('title')->take(50)->toArray();

        return response($brands);
    }
}
