<?php

namespace App\Http\Controllers\Admin\Products;

use App\Http\Controllers\Controller;
use App\Http\Requests\Admin\ProductCommentEditRequest;
use App\Models\ProductComment;
use App\Services\Admin\Products\ProductCommentsService;
use Illuminate\Http\Request;
use Inertia\Inertia;

class CommentsController extends Controller
{
    /**
     * product comment list
     */
    public function index(Request $request, ProductCommentsService $productCommentsService)
    {
        return Inertia::render('Admin/Products/Comments/List', [
            'comments' => $productCommentsService->getComments($request)
        ]);
    }

    /**
     * accept comment
     */
    public function accept(ProductComment $comment)
    {
        if ($comment->status == 'awaiting') {
            $comment->update([
                'status' => 'publish',
            ]);

            /** Report Action */
            $this->report(__("messages.report_product_comment_accepted"), 'product', $comment->product);

            return redirect()->back()->with('message', [__('messages.comment_accepted')]);
        }

        return redirect()->back();
    }

    /**
     * accept comment
     */
    public function remove_report(ProductComment $comment)
    {
        $comment->update([
            'reported' => false,
        ]);

        /** Report Action */
        $this->report(__("messages.report_product_comment_report_check"), 'product', $comment->product);

        return redirect()->back()->with('message', [__('messages.comment_report_removed')]);
    }

    /**
     * edit comment
     */
    public function edit(ProductCommentEditRequest $request, ProductComment $comment)
    {
        // validate buy suggest
        $buy_suggest = $comment->buy_suggest;
        if ($comment->buy_suggest != null) {
            $buy_suggest = $request->exists('buy_suggest') && $request->buy_suggest ? '1' : '0';
        }

        $comment->update([
            'content' => $request->content,
            'rating' => $request->rating,
            'vote_up' => $request->exists('vote_up') && $request->vote_up != null ? $request->vote_up : $comment->vote_up,
            'vote_down' => $request->exists('vote_down') && $request->vote_down != null ? $request->vote_down : $comment->vote_down,
            'buy_suggest' => $buy_suggest,
            'status' => 'publish',
        ]);

        /** Report Action */
        $this->report(__("messages.report_product_comment_edited"), 'product', $comment->product);

        return redirect()->back()->with('message', [__('messages.comment_edited')]);
    }

    /**
     * send comment answer
     */
    public function send_answer(Request $request, ProductComment $comment)
    {
        $request->validate([
            'content' => 'required'
        ]);

        /** @var App\Models\User $user */
        $user = auth()->user();

        $answer = [
            'user' => [
                'name' => $user->get_name,
                'role' => $user->role,
            ],
            'content' => $request->content,
        ];

        $comment->update([
            'answer' => serialize($answer),
            'status' => 'publish'
        ]);

        /** Report Action */
        $this->report(__("messages.report_product_comment_send_answer"), 'product', $comment->product);

        return redirect()->back()->with('message', [__('messages.comment_answer_send')]);
    }

    /**
     * delete comment answer
     */
    public function delete_answer(ProductComment $comment)
    {
        $comment->update([
            'answer' => null,
        ]);

        /** Report Action */
        $this->report(__("messages.report_product_comment_answer_deleted"), 'product', $comment->product);

        return redirect()->back()->with('message', [__('messages.comment_answer_deleted')]);
    }

    /**
     * delete comment
     */
    public function delete(ProductComment $comment)
    {
        $comment->delete();

        /** Report Action */
        $this->report(__("messages.report_product_comment_deleted"), 'product', $comment->product);

        return redirect()->back()->with('message', [__('messages.comment_deleted')]);
    }
}
