<?php

namespace App\Http\Controllers\Admin\Stories;

use App\Helpers\DropdownListHelper;
use App\Http\Controllers\Controller;
use App\Http\Requests\Admin\CreateStoryRequest;
use App\Models\Setting;
use App\Models\Story;
use App\Services\Admin\Stories\StoriesService;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Inertia\Inertia;
use Morilog\Jalali\Jalalian;

class StoriesController extends Controller
{
    use DropdownListHelper;

    /**
     * Display a listing of the resource.
     */
    public function index(Request $request, StoriesService $storiesService)
    {
        // stories status
        $storiesStatus = Setting::where('key', 'stories_status')->first();
        $storiesStatus = $storiesStatus != null ? $storiesStatus->value : false;
        $storiesStatus = $storiesStatus == "1" ? true : false;

        return Inertia::render('Admin/Stories/List', [
            'status' => $storiesStatus,
            'stories' => $storiesService->getStories($request),
        ]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return Inertia::render('Admin/Stories/Create', [
            'storyTypes' => $this->formStoryTypeList(),
        ]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(CreateStoryRequest $request)
    {
        /** @var \App\Models\User $user */
        $user = auth()->user();

        $expiredAt = Jalalian::fromFormat('Y/m/d', $request->expired_at)->getTimestamp();
        $expiredAt = Jalalian::forge($expiredAt)->toCarbon()->setTimezone('Asia/Tehran');
        $maxDate = Carbon::create(2032, 12, 31, 23, 59, 59, 'Asia/Tehran');

        if ($expiredAt->greaterThan($maxDate)) {
            return redirect()->back()->withErrors([__('messages.the_date_should_not_be_older_than_1410')]);
        }
        $expiredAt = $expiredAt->format('Y-m-d H:i:s');

        $widgets = $request->widgets;
        if (isset($widgets['link']['title'])) {
            $widgets['link']['title'] = mb_convert_encoding(urldecode($widgets['link']['title']), 'UTF-8', 'UTF-8');
        }
        if (isset($widgets['link']['url'])) {
            $widgets['link']['url'] = mb_convert_encoding(urldecode($widgets['link']['url']), 'UTF-8', 'UTF-8');
        }

        $user->stories()->create([
            'type' => $request->type,
            'title' => $request->title,
            'cover' => $request->cover,
            'file' => $request->file,
            'widgets' => $request->exists('widgets') ? serialize($widgets) : null,
            'status' => 'publish',
            'expired_at' => $expiredAt,
        ]);

        return redirect()->route('admin.stories.index')->with('message', [__('messages.story_added')]);
    }

    /**
     * Display the specified resource.
     */
    public function show(Story $story)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Story $story)
    {
        return Inertia::render('Admin/Stories/Edit', [
            'story' => $story,
            'storyTypes' => $this->formStoryTypeList(),
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(CreateStoryRequest $request, Story $story)
    {
        /** @var \App\Models\User $user */
        $user = auth()->user();

        $expiredAt = Jalalian::fromFormat('Y/m/d', $request->expired_at)->getTimestamp();
        $expiredAt = Jalalian::forge($expiredAt)->toCarbon()->setTimezone('Asia/Tehran');
        $maxDate = Carbon::create(2032, 12, 31, 23, 59, 59, 'Asia/Tehran');
        if ($expiredAt->greaterThan($maxDate)) {
            return redirect()->back()->withErrors([__('messages.the_date_should_not_be_older_than_1410')]);
        }
        $expiredAt = $expiredAt->format('Y-m-d H:i:s');

        $widgets = $request->widgets;
        if (isset($widgets['link']['title'])) {
            $widgets['link']['title'] = mb_convert_encoding(urldecode($widgets['link']['title']), 'UTF-8', 'UTF-8');
        }
        if (isset($widgets['link']['url'])) {
            $widgets['link']['url'] = mb_convert_encoding(urldecode($widgets['link']['url']), 'UTF-8', 'UTF-8');
        }

        $story->update([
            'type' => $request->type,
            'title' => $request->title,
            'cover' => $request->cover,
            'file' => $request->file,
            'widgets' => $request->exists('widgets') ? serialize($widgets) : null,
            'expired_at' => $expiredAt,
        ]);

        return redirect()->route('admin.stories.index')->with('message', [__('messages.story_edited')]);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Story $story)
    {
        $story->delete();

        return redirect()->route('admin.stories.index')->with('message', [__('messages.story_deleted')]);
    }

    /**
     * change stories status
     */
    public function status()
    {
        // stories status
        $storiesStatus = Setting::where('key', 'stories_status')->first();
        $storiesStatus = $storiesStatus != null ? $storiesStatus->value : false;
        $storiesStatus = $storiesStatus == "1" ? true : false;

        if ($storiesStatus) {
            Setting::updateOrCreate(['key' => 'stories_status'], [
                'value' => false
            ]);

            return redirect()->route('admin.stories.index')->with('message', [__('messages.stories_deactivated')]);
        } else {
            Setting::updateOrCreate(['key' => 'stories_status'], [
                'value' => true
            ]);

            return redirect()->route('admin.stories.index')->with('message', [__('messages.stories_activated')]);
        }
    }
}
