<?php

namespace App\Http\Controllers\Api\v1\Inventories;

use App\Helpers\UUIDHelper;
use App\Http\Controllers\Controller;
use App\Models\File;
use App\Models\Inventory;
use App\Models\Product;
use App\Models\Storage as ModelsStorage;
use App\Models\User;
use App\Services\Admin\Storage\InventoriesService;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Storage;
use Intervention\Image\Drivers\Gd\Driver;
use Intervention\Image\Drivers\Gd\Encoders\WebpEncoder;
use Intervention\Image\ImageManager;
use Morilog\Jalali\Jalalian;

class InventoriesController extends Controller
{
    use UUIDHelper;

    /**
     * product inventories
     */
    public function index(Request $request, $product_uuid)
    {
        $perPage = $request->exists('per_page') && is_int((int)$request->per_page) && (int)$request->per_page > 0 ? (int)$request->per_page : 100;

        // product
        $product = Product::where('uuid', $product_uuid)->first();
        if ($product == null || $product->status == 'deleted' || $product->status == 'deep_deleted') {
            return response()->json([
                'success' => false,
                'message' => __('messages.product_not_found'),
                'code' => 404,
            ]);
        }

        // product inventories
        $inventories = $product->inventories()->orderByDesc('created_at')->paginate($perPage)->through(fn($inventory) => [
            'id' => $inventory->id,
            'uuid' => $inventory->uuid,
            'variables' => $inventory->props,
            'price' => $inventory->price,
            'discount_price' => $inventory->discount_price,
            'discount_expire' => $inventory->discount_expire != null ? $inventory->jalali_discount_expire[1] : null,
            'discount_tree' => $inventory->get_discount_tree,
            'count' => $inventory->count,
            'min_sale' => $inventory->min_sale,
            'max_sale' => $inventory->max_sale,
            'original' => $inventory->original,
            'used' => $inventory->used,
            'weight' => $inventory->weight,
            'image' => $inventory->image != null ? [
                'url' => $inventory->get_image['url'],
                'alt' => $inventory->get_image['alt']
            ] : null,
            'price_changes' => $inventory->get_price_changes_chart,
            'send_time' => $inventory->send_time,
            'status' => $inventory->status,
            'created_at' => $inventory->created_at,
            'updated_at' => $inventory->updated_at,
            'store' => $inventory->store ? [
                'id' => $inventory->store->id,
                'uuid' => $inventory->store->uuid,
                'user_id' => $inventory->store->user_id,
                'name' => $inventory->store->name,
                'logo' => $inventory->store->get_logo,
                'type' => $inventory->store->type,
                'direct_post' => $inventory->store->direct_post,
                'meta' => $inventory->store->get_meta,
                'status' => $inventory->store->status,
                'created_at' => $inventory->store->created_at,
                'updated_at' => $inventory->store->updated_at,
            ] : null,
            'storage' => $inventory->storage ? [
                'id' => $inventory->storage->id,
                'uuid' => $inventory->storage->uuid,
                'name' => $inventory->storage->name,
                'address' => $inventory->storage->address,
                'product_count' => $inventory->storage->product_count,
                'created_at' => $inventory->storage->created_at,
                'updated_at' => $inventory->storage->updated_at,
            ] : null,
            'product' => $inventory->product ? [
                'id' => $inventory->product->id,
                'uuid' => $inventory->product->uuid,
                'title' => $inventory->product->title,
                'slug' => $inventory->product->slug,
                'variables' => $inventory->product->price_model,
                'commission' => $inventory->product->commission,
                'created_at' => $inventory->product->created_at,
                'updated_at' => $inventory->product->updated_at,
            ] : null,
        ]);

        return response()->json([
            'success' => true,
            'response' => $inventories,
            'code' => 200,
        ]);
    }

    /**
     * inventory details
     */
    public function show(Request $request, $inventory_uuid)
    {
        // inventory
        $inventory = Inventory::where('uuid', $inventory_uuid)->with(['product', 'store', 'storage'])->first();
        if ($inventory == null) {
            return response()->json([
                'success' => false,
                'message' => __('messages.inventory_not_found'),
                'code' => 404,
            ]);
        }

        $inventory = [
            'id' => $inventory->id,
            'uuid' => $inventory->uuid,
            'variables' => $inventory->props,
            'price' => $inventory->price,
            'discount_price' => $inventory->discount_price,
            'discount_expire' => $inventory->discount_expire,
            'discount_tree' => $inventory->get_discount_tree,
            'count' => $inventory->count,
            'min_sale' => $inventory->min_sale,
            'max_sale' => $inventory->max_sale,
            'original' => $inventory->original,
            'used' => $inventory->used,
            'weight' => $inventory->weight,
            'image' => $inventory->image != null ? [
                'url' => $inventory->get_image['url'],
                'alt' => $inventory->get_image['alt']
            ] : null,
            'price_changes' => $inventory->get_price_changes_chart,
            'send_time' => $inventory->send_time,
            'status' => $inventory->status,
            'created_at' => $inventory->created_at,
            'updated_at' => $inventory->updated_at,
            'store' => $inventory->store ? [
                'id' => $inventory->store->id,
                'uuid' => $inventory->store->uuid,
                'user_id' => $inventory->store->user_id,
                'name' => $inventory->store->name,
                'logo' => $inventory->store->get_logo,
                'type' => $inventory->store->type,
                'direct_post' => $inventory->store->direct_post,
                'meta' => $inventory->store->get_meta,
                'status' => $inventory->store->status,
                'created_at' => $inventory->store->created_at,
                'updated_at' => $inventory->store->updated_at,
            ] : null,
            'storage' => $inventory->storage ? [
                'id' => $inventory->storage->id,
                'uuid' => $inventory->storage->uuid,
                'name' => $inventory->storage->name,
                'address' => $inventory->storage->address,
                'product_count' => $inventory->storage->product_count,
                'created_at' => $inventory->storage->created_at,
                'updated_at' => $inventory->storage->updated_at,
            ] : null,
            'product' => $inventory->product ? [
                'id' => $inventory->product->id,
                'uuid' => $inventory->product->uuid,
                'title' => $inventory->product->title,
                'slug' => $inventory->product->slug,
                'variables' => $inventory->product->price_model,
                'commission' => $inventory->product->commission,
                'created_at' => $inventory->product->created_at,
                'updated_at' => $inventory->product->updated_at,
            ] : null,
        ];

        return response()->json([
            'success' => true,
            'response' => $inventory,
            'code' => 200,
        ]);
    }

    /**
     * create inventory
     */
    public function create(Request $request, $product_uuid, InventoriesService $inventoriesService)
    {
        $request->validate([
            'user_id' => 'required',
            'storage_uuid' => 'required',
            'price' => 'required',
            'count' => 'required',
        ]);

        /** user */
        $user = User::find($request->user_id);
        if ($user == null) {
            return response()->json([
                'success' => false,
                'response' => __('messages.user_not_found'),
                'code' => 404,
            ]);
        } else if ($user->type != 'owner' && $user->type != 'admin') {
            return response()->json([
                'success' => false,
                'response' => __('messages.user_not_admin'),
                'code' => 403,
            ]);
        }

        // product
        $product = Product::where('uuid', $product_uuid)->first();
        if ($product == null || $product->status == 'deleted' || $product->status == 'deep_deleted') {
            return response()->json([
                'success' => false,
                'message' => __('messages.product_not_found'),
                'code' => 404,
            ]);
        }

        // init fields
        $props = $inventoriesService->getVariables($request, $product);
        $price = (int)str_replace(',', '', $request->price);
        $purchasePrice = $request->has('purchase_price') ? (int)str_replace(',', '', $request->purchase_price) : null;
        $discount_price = null;
        $discount_expire = null;
        $count = (int)str_replace(',', '', $request->count);
        $minSale = $request->exists('min_sale') && $request->min_sale != null && $request->min_sale > 0 ? (int)str_replace(',', '', $request->min_sale) : 1;
        $maxSale = $request->exists('max_sale') && $request->max_sale != null && $request->max_sale != 0 ? (int)str_replace(',', '', $request->max_sale) : null;
        $discountTree = [];

        // validate single price
        if ($inventoriesService->validateSinglePrice($product))
            return response()->json([
                'success' => false,
                'message' => __('messages.just_can_add_one_price'),
                'code' => 400,
            ]);

        if (count($props) > 0) {
            // validate props
            if ($inventoriesService->validateProps($props, $product) !== false)
                $props = $inventoriesService->validateProps($props, $product);
            else
                return response()->json([
                    'success' => false,
                    'message' => __('messages.product_props_required'),
                    'code' => 400,
                ]);

            // validate duplicated props
            if (!$inventoriesService->validateDuplicateProps($props, $product))
                return response()->json([
                    'success' => false,
                    'message' => __('messages.price_is_duplicated'),
                    'code' => 400,
                ]);
        }

        // validate discount fields
        if ($request->discount_price != null && $request->discount_price != 0) {
            // compare discount_price and main price
            $discount_price = (int)str_replace(',', '', $request->discount_price);
            if ($discount_price > $price) {
                return response()->json([
                    'success' => false,
                    'message' => __('messages.discount_price_larger_than_price'),
                    'code' => 400,
                ]);
            }

            // check discount_expire required field
            if ($request->discount_expire != null) {
                $discount_expire = Jalalian::fromFormat('Y/m/d', $request->discount_expire)->getTimestamp();
                $discount_expire = Jalalian::forge($discount_expire)
                    ->toCarbon()
                    ->setTimezone('Asia/Tehran')
                    ->setTime(23, 59, 59); // تنظیم ساعت به 23:59:00
                $discount_expire = $discount_expire->format('Y-m-d H:i:s');
            }
        }

        // validate discount tree
        if ($request->exists('discount_tree') && is_array($request->discount_tree) && count($request->discount_tree) > 0) {
            if ($request->discount_price == null) {
                return response()->json([
                    'success' => false,
                    'message' => __('messages.to_add_discount_tree_first_fill_main_discount'),
                    'code' => 400,
                ]);
            }

            foreach ($request->discount_tree as $dt) {
                // check for null
                if ($dt['count'] != null && $dt['count'] != '' && $dt['price'] != null && $dt['price'] != '') {

                    // discount_tree fields
                    $dtCount = (int)str_replace(',', '', $dt['count']);
                    $dtPrice = (int)str_replace(',', '', $dt['price']);

                    // compare discount_tree price and main price
                    if ($dtPrice > $price) {
                        return response()->json([
                            'success' => false,
                            'message' => __('messages.discount_tree_price_larger_than_price'),
                            'code' => 400,
                        ]);
                    }
                    if ($discount_price != null && $dtPrice > $discount_price) {
                        return response()->json([
                            'success' => false,
                            'message' => __('messages.discount_tree_price_larger_than_discount_price'),
                            'code' => 400,
                        ]);
                    }

                    $discountTree[] = [
                        'count' => $dtCount,
                        'price' => $dtPrice,
                    ];
                }
            }

            // sort discount tree
            sort($discountTree);

            // validate discount tree counts
            $discountTreeCounts = [];
            foreach ($discountTree as $dt) {
                $discountTreeCounts[$dt['count']][] = $dt['count'];
            }
            if (count($discountTreeCounts) != count($discountTree)) {
                return response()->json([
                    'success' => false,
                    'message' => __('messages.discount_tree_count_wrong'),
                    'code' => 400,
                ]);
            }

            // validate discount tree prices
            $dtPrevPrice = null;
            foreach ($discountTree as $dt) {
                if ($dtPrevPrice != null) {
                    if ($dtPrevPrice <= $dt['price']) {
                        return response()->json([
                            'success' => false,
                            'message' => __('messages.discount_tree_price_wrong'),
                            'code' => 400,
                        ]);
                    }
                }
                $dtPrevPrice = $dt['price'];
            }
        }

        // check min and max sale value
        if ($maxSale != null && $maxSale != '' && $minSale > $maxSale) {
            return response()->json([
                'success' => false,
                'message' => __('messages.min_sale_larger_than_max_sale'),
                'code' => 400,
            ]);
        }
        if ($minSale != null && $count != 0 && $minSale > $count) {
            return response()->json([
                'success' => false,
                'message' => __('messages.min_sale_larger_than_count'),
                'code' => 400,
            ]);
        }

        // validate weight
        $weight = null;
        if ($request->exists('weight') && $request->weight != null && is_numeric(str_replace(',', '', $request->weight))) {
            $weight = str_replace(',', '', $request->weight);
        }

        // price changes
        $priceChanges[Carbon::now()->timestamp] = $discount_price != null ? $discount_price : $price;

        /** images */
        $image = null;
        if ($request->has('image')) {
            $imageUrl = $request->image['src'] ?? null;
            $imageAlt = $request->image['alt'] ?? null;
            if ($imageUrl) {
                $fileName = basename(parse_url($imageUrl, PHP_URL_PATH));
                $file = File::where('name', $fileName)->first();
                if ($file) {
                    $image = $file->id;
                } else {
                    $uploadedFile = $this->uploadFromUrl($imageUrl, $imageAlt, $user);
                    if ($uploadedFile && isset($uploadedFile['id'])) {
                        $image = $uploadedFile['id'];
                    }
                }
            }
        }

        // validate storage
        $storage = ModelsStorage::where('uuid', $request->storage_uuid)->first();
        if ($storage == null) {
            return response()->json([
                'success' => false,
                'message' => __('messages.storage_not_found'),
                'code' => 400,
            ]);
        }

        // add inventory
        $inventory = $product->inventories()->create([
            'storage_id' => $storage->id,
            'uuid' => $this->generateUniqueRandomNumber(8, \App\Models\Product::class, 'uuid', 'inv-'),
            'price' => $price,
            'discount_price' => $discount_price,
            'discount_expire' => $discount_expire,
            'count' => $count,
            'min_sale' => $minSale,
            'max_sale' => $maxSale,
            'original' => $request->has('original') ? $request->original : true,
            'used' => $request->has('used') ?  $request->used : false,
            'weight' => $weight,
            'purchase_price' => $purchasePrice,
            'discount_tree' => is_array($discountTree) && count($discountTree) > 0 ? serialize($discountTree) : null,
            'price_changes' => $priceChanges != null ? serialize($priceChanges) : null,
            'image' => $image,
            'send_time' => $request->exists('send_time') ? $request->send_time : null,
            'status' => 'publish',
        ]);

        // add inventory props
        $pmID = 0;
        foreach ($product->price_model as $key => $pm) {
            if (array_key_exists($key, $props)) {
                if (array_key_exists('value', $props[$key])) {
                    if ($props[$key]['value'] == null || $props[$key]['value'] == '') {
                        return response()->json([
                            'success' => false,
                            'message' => __('messages.product_props_required'),
                            'code' => 400,
                        ]);
                    }

                    $pmModel = $inventory->inventoryProps()->create([
                        'child' => $pmID,
                        'name' => $pm['name'],
                        'type' => $pm['type'],
                        'value' => $pm['type'] == 'color' ? serialize($props[$key]['value']) : $props[$key]['value'],
                    ]);
                    $pmID = $pmModel->id;
                } else {
                    return response()->json([
                        'success' => false,
                        'message' => __('messages.product_props_required'),
                        'code' => 400,
                    ]);
                }
            } else {
                return response()->json([
                    'success' => false,
                    'message' => __('messages.product_props_required'),
                    'code' => 400,
                ]);
            }
        }

        /** product notifications */
        $inventoriesService->handleProductNotifications($product);

        return response()->json([
            'success' => true,
            'message' => __('messages.inventory_created'),
            'inventory_uuid' => $inventory->uuid,
            'code' => 200,
        ]);
    }

    /**
     * update inventory
     */
    public function update(Request $request, $inventory_uuid, InventoriesService $inventoriesService)
    {
        $request->validate([
            'user_id' => 'required',
        ]);

        /** user */
        $user = User::find($request->user_id);
        if ($user == null) {
            return response()->json([
                'success' => false,
                'response' => __('messages.user_not_found'),
                'code' => 404,
            ]);
        } else if ($user->type != 'owner' && $user->type != 'admin') {
            return response()->json([
                'success' => false,
                'response' => __('messages.user_not_admin'),
                'code' => 403,
            ]);
        }

        // inventory
        $inventory = Inventory::where('uuid', $inventory_uuid)->first();
        if ($inventory == null) {
            return response()->json([
                'success' => false,
                'message' => __('messages.inventory_not_found'),
                'code' => 404,
            ]);
        }

        // product
        $product = $inventory->product;
        if ($product == null || $product->status == 'deleted' || $product->status == 'deep_deleted') {
            return response()->json([
                'success' => false,
                'message' => __('messages.product_not_found'),
                'code' => 404,
            ]);
        }

        // init fields
        $props = $inventoriesService->getVariables($request, $product);
        $price = (int)str_replace(',', '', $request->price);
        $purchasePrice = $request->has('purchase_price') ? (int)str_replace(',', '', $request->purchase_price) : null;
        $discount_price = null;
        $discount_expire = null;
        $count = (int)str_replace(',', '', $request->count);
        $minSale = $request->exists('min_sale') && $request->min_sale != null && $request->min_sale > 0 ? (int)str_replace(',', '', $request->min_sale) : 1;
        $maxSale = $request->exists('max_sale') && $request->max_sale != null && $request->max_sale != 0 ? (int)str_replace(',', '', $request->max_sale) : null;
        $discountTree = [];

        if (count($props) > 0) {
            // validate props
            if ($inventoriesService->validateProps($props, $product, $inventory) != false)
                $props = $inventoriesService->validateProps($props, $product, $inventory);
            else
                return response()->json([
                    'success' => false,
                    'message' => __('messages.product_props_required'),
                    'code' => 400,
                ]);

            // validate duplicated props
            if (!$inventoriesService->validateDuplicateProps($props, $product, $inventory))
                return response()->json([
                    'success' => false,
                    'message' => __('messages.price_is_duplicated'),
                    'code' => 400,
                ]);
        }

        // validate discount fields
        if ($request->discount_price != null && $request->discount_price != 0) {
            // compare discount_price and main price
            $discount_price = (int)str_replace(',', '', $request->discount_price);
            if ($discount_price > $price) {
                return response()->json([
                    'success' => false,
                    'message' => __('messages.discount_price_larger_than_price'),
                    'code' => 400,
                ]);
            }

            // check discount_expire required field
            if ($request->discount_expire != null) {
                $discount_expire = Jalalian::fromFormat('Y/m/d', $request->discount_expire)->getTimestamp();
                $discount_expire = Jalalian::forge($discount_expire)
                    ->toCarbon()
                    ->setTimezone('Asia/Tehran')
                    ->setTime(23, 59, 59); // تنظیم ساعت به 23:59:00
                $discount_expire = $discount_expire->format('Y-m-d H:i:s');
            }
        }

        // validate discount tree
        if ($request->exists('discount_tree') && is_array($request->discount_tree) && count($request->discount_tree) > 0) {
            if ($request->discount_price == null) {
                return response()->json([
                    'success' => false,
                    'message' => __('messages.to_add_discount_tree_first_fill_main_discount'),
                    'code' => 400,
                ]);
            }

            foreach ($request->discount_tree as $dt) {
                // check for null
                if ($dt['count'] != null && $dt['count'] != '' && $dt['price'] != null && $dt['price'] != '') {

                    // discount_tree fields
                    $dtCount = (int)str_replace(',', '', $dt['count']);
                    $dtPrice = (int)str_replace(',', '', $dt['price']);

                    // compare discount_tree price and main price
                    if ($dtPrice > $price) {
                        return response()->json([
                            'success' => false,
                            'message' => __('messages.discount_tree_price_larger_than_price'),
                            'code' => 400,
                        ]);
                    }
                    if ($discount_price != null && $dtPrice > $discount_price) {
                        return response()->json([
                            'success' => false,
                            'message' => __('messages.discount_tree_price_larger_than_discount_price'),
                            'code' => 400,
                        ]);
                    }

                    $discountTree[] = [
                        'count' => $dtCount,
                        'price' => $dtPrice,
                    ];
                }
            }

            // sort discount tree
            sort($discountTree);

            // validate discount tree counts
            $discountTreeCounts = [];
            foreach ($discountTree as $dt) {
                $discountTreeCounts[$dt['count']][] = $dt['count'];
            }
            if (count($discountTreeCounts) != count($discountTree)) {
                return response()->json([
                    'success' => false,
                    'message' => __('messages.discount_tree_count_wrong'),
                    'code' => 400,
                ]);
            }

            // validate discount tree prices
            $dtPrevPrice = null;
            foreach ($discountTree as $dt) {
                if ($dtPrevPrice != null) {
                    if ($dtPrevPrice <= $dt['price']) {
                        return response()->json([
                            'success' => false,
                            'message' => __('messages.discount_tree_price_wrong'),
                            'code' => 400,
                        ]);
                    }
                }
                $dtPrevPrice = $dt['price'];
            }
        }

        // check min and max sale value
        if ($maxSale != null && $maxSale != '' && $minSale > $maxSale) {
            return response()->json([
                'success' => false,
                'message' => __('messages.min_sale_larger_than_max_sale'),
                'code' => 400,
            ]);
        }
        if ($minSale != null && $count != 0 && $minSale > $count) {
            return response()->json([
                'success' => false,
                'message' => __('messages.min_sale_larger_than_count'),
                'code' => 400,
            ]);
        }

        // validate weight
        $weight = null;
        if ($request->exists('weight') && $request->weight != null && is_numeric(str_replace(',', '', $request->weight))) {
            $weight = str_replace(',', '', $request->weight);
        }

        /** images */
        $image = null;
        if ($request->has('image')) {
            $imageUrl = $request->image['src'] ?? null;
            $imageAlt = $request->image['alt'] ?? null;
            if ($imageUrl) {
                $fileName = basename(parse_url($imageUrl, PHP_URL_PATH));
                $file = File::where('name', $fileName)->first();
                if ($file) {
                    $image = $file->id;
                } else {
                    $uploadedFile = $this->uploadFromUrl($imageUrl, $imageAlt, $user);
                    if ($uploadedFile && isset($uploadedFile['id'])) {
                        $image = $uploadedFile['id'];
                    }
                }
            }
        }

        // validate storage
        $storage = $inventory->storage;
        if ($request->has('storage_uuid')) {
            $storage = ModelsStorage::where('uuid', $request->storage_uuid)->first();
        }
        if ($storage == null) {
            return response()->json([
                'success' => false,
                'message' => __('messages.storage_not_found'),
                'code' => 400,
            ]);
        }

        // update inventory
        $inventory->update([
            'storage_id' => $storage->id,
            'price' => $request->has('price') ? $price : $inventory->price,
            'discount_price' => $request->has('discount_price') ? $discount_price : $inventory->discount_price,
            'discount_expire' => $request->has('discount_expire') ? $discount_expire : $inventory->discount_expire,
            'count' => $request->has('count') ? $count : $inventory->count,
            'min_sale' => $request->has('min_sale') ? $minSale : $inventory->min_sale,
            'max_sale' => $request->has('max_sale') ? $maxSale : $inventory->max_sale,
            'original' => $request->has('original') ? $request->original : $inventory->original,
            'used' => $request->has('used') ?  $request->used :  $inventory->used,
            'weight' => $request->has('weight') ? $weight : $inventory->weight,
            'purchase_price' => $purchasePrice,
            'discount_tree' => is_array($discountTree) && count($discountTree) > 0 ? serialize($discountTree) : null,
            'price_changes' => serialize($inventoriesService->handlePriceChanges($inventory->get_price_changes, $discount_price != null ? $discount_price : $price)),
            'image' => $image,
            'send_time' => $request->has('send_time') ? $request->send_time : $inventory->send_time,
            'status' => $request->has('status') ? $request->status : $inventory->status,
        ]);

        // update inventory props
        $inventoryProps = $inventory->inventoryProps()->get();
        foreach ($inventoryProps as $inventoryProp) {
            if (array_key_exists($inventoryProp->id, $props)) {
                if (array_key_exists('value', $props[$inventoryProp->id])) {
                    if ($props[$inventoryProp->id]['value'] == null || $props[$inventoryProp->id]['value'] == '') {
                        return response()->json([
                            'success' => false,
                            'message' => __('messages.product_props_required'),
                            'code' => 400,
                        ]);
                    }

                    $inventoryProp->update([
                        'value' => $inventoryProp->type == 'color' ? serialize($props[$inventoryProp->id]['value']) : $props[$inventoryProp->id]['value'],
                    ]);
                } else {
                    return response()->json([
                        'success' => false,
                        'message' => __('messages.product_props_required'),
                        'code' => 400,
                    ]);
                }
            } else {
                return response()->json([
                    'success' => false,
                    'message' => __('messages.product_props_required'),
                    'code' => 400,
                ]);
            }
        }

        /** product notifications */
        $inventoriesService->handleProductNotifications($product);

        return response()->json([
            'success' => true,
            'message' => __('messages.inventory_updated'),
            'inventory_uuid' => $inventory->uuid,
            'code' => 200,
        ]);
    }

    /**
     * delete inventory
     */
    public function delete(Request $request, $inventory_uuid)
    {
        $inventory = Inventory::where('uuid', $inventory_uuid)->first();

        if ($inventory == null) {
            return response()->json([
                'success' => false,
                'message' => __('messages.inventory_not_found'),
                'code' => 404,
            ]);
        }

        // delete inventory from users cart
        $inventory->consignmentItems()->whereHas('consignment', function ($query) {
            $query->whereHas('order', function ($query) {
                $query->whereDoesntHave('orderGroup');
            });
        })->delete();

        // delete
        $inventory->delete();

        return response()->json([
            'success' => true,
            'message' => __('messages.inventory_deleted'),
            'code' => 200,
        ]);
    }

    /**
     * upload image from url
     */
    private function uploadFromUrl($url, $alt, $user)
    {
        try {
            // دانلود فایل با timeout
            $response = Http::timeout(2)->get($url);

            $imageContent = $response->body();

            // ذخیره موقت
            $tmpFile = tempnam(sys_get_temp_dir(), 'img');
            file_put_contents($tmpFile, $imageContent);

            // گرفتن پسوند اصلی
            $fileInfo = pathinfo(parse_url($url, PHP_URL_PATH));
            $originalExtension = strtolower($fileInfo['extension'] ?? 'jpg');
            $fileType = match ($originalExtension) {
                'jpeg', 'jpg', 'png', 'webp', 'gif', 'svg', 'avif' => 'image',
                default => null,
            };

            if ($fileType !== 'image') {
                return null;
            }

            // پردازش تصویر
            $manager = ImageManager::withDriver(new Driver());
            $imageTemp = $manager->read($tmpFile);
            $imageResolution = $imageTemp->width() . '×' . $imageTemp->height();

            // تبدیل به webp و فشرده‌سازی
            $optimizationPercent = 30;
            $finalImageTemp = $imageTemp->encode(new WebpEncoder($optimizationPercent))->__toString();
            $fileFormat = 'webp';

            // مسیر ذخیره‌سازی
            $fileName = 'file.' . time() . '.' . rand(10000, 99999) . '.' . $fileFormat;
            $fileSaveAddressNoName = 'library/' . date('Y-m-d') . '/' . $user->id;
            $fileSaveAddressImage = $fileSaveAddressNoName . '/' . $fileName;

            $uploadResult = Storage::disk('media')->put($fileSaveAddressImage, $finalImageTemp, 'public');

            if ($uploadResult) {
                return $user->files()->create([
                    'name'        => $fileName,
                    'description' => $alt,
                    'url'         => '/media/' . $fileSaveAddressImage,
                    'extension'   => $fileFormat,
                    'size'        => strlen($finalImageTemp) / 1000,
                    'resolution'  => $imageResolution,
                    'type'        => $fileType,
                ])->toArray();
            }

            return null;
        } catch (\Exception $e) {
            return null;
        }
    }
}
