<?php

namespace App\Http\Controllers\Api\v1\Storages;

use App\Helpers\UUIDHelper;
use App\Http\Controllers\Controller;
use App\Models\Product;
use App\Models\Storage;
use Illuminate\Http\Request;

class StoragesController extends Controller
{
    use UUIDHelper;

    /**
     * storage list
     */
    public function index(Request $request)
    {
        $perPage = $request->exists('per_page') && is_int((int)$request->per_page) && (int)$request->per_page > 0 ? (int)$request->per_page : 20;

        $storages = Storage::where('store_id', null)->orderBy('created_at', 'desc')->paginate($perPage)->through(fn($storage) => [
            'id' => $storage->id,
            'uuid' => $storage->uuid,
            'name' => $storage->name,
            'address' => $storage->address,
            'created_at' => $storage->created_at,
            'updated_at' => $storage->updated_at,
        ]);

        return response()->json([
            'success' => true,
            'response' => $storages,
            'code' => 200,
        ]);
    }

    /**
     * storage products
     */
    public function show(Request $request, $storage_uuid)
    {
        $perPage = $request->exists('per_page') && is_int((int)$request->per_page) && (int)$request->per_page > 0 ? (int)$request->per_page : 100;

        // storage
        $storage = Storage::where('uuid', $storage_uuid)->first();
        if ($storage == null) {
            return response()->json([
                'success' => false,
                'message' => __('messages.storage_not_found'),
                'code' => 404,
            ]);
        }

        $products = Product::with([
            'brand',
            'productCategory',
            'inventories',
            'inventories.store',
            'inventories.storage'
        ])->whereHas('inventories', function ($query) use ($storage) {
            $query->where('storage_id', $storage->id);
        })->where('status', '!=', 'deep_deleted')->where('status', '!=', 'deleted')->when($request->input('status'), function ($query, $status) {
            $query->where('status', $status);
        })->orderByDesc('created_at')->paginate($perPage)->through(fn($product) => [
            'id' => $product->id,
            'uuid' => $product->uuid,
            'user_id' => $product->user_id,
            'edited_by' => $product->edited_by,
            'brand' => $product->brand ? [
                'id' => $product->brand->id,
                'title' => $product->brand->title,
                'slug' => $product->brand->slug,
                'logo' => $product->brand->get_logo,
                'status' => $product->brand->status,
                'created_at' => $product->brand->created_at,
                'updated_at' => $product->brand->updated_at,
            ] : null,
            'category' => $product->productCategory ? [
                'id' => $product->productCategory->id,
                'title' => $product->productCategory->title,
                'slug' => $product->productCategory->slug,
                'image' => $product->productCategory->get_image,
                'status' => $product->productCategory->status,
                'created_at' => $product->productCategory->created_at,
                'updated_at' => $product->productCategory->updated_at,
            ] : null,
            'title' => $product->title,
            'title_en' => $product->title_en,
            'content' => $product->content,
            'warnings' => $product->warnings,
            'tabs' => $product->get_tabs,
            'guide' => $product->guide,
            'images' => $product->get_all_images,
            'videos' => $product->get_videos,
            'model_3d' => $product->model_3d,
            'attrs' => $product->get_attrs,
            'specifications' => $product->get_specifications,
            'seo_title' => $product->seo_title,
            'seo_description' => $product->seo_description,
            'seo_keywords' => json_decode($product->seo_keywords),
            'seo_canonical' => $product->seo_canonical,
            'view_count' => $product->view_count,
            'in_stock_status' => $product->in_stock_status,
            'inquiry_options' => $product->get_inquiry_options,
            'variables' => $product->raw_price_model,
            'inventories' => $product->inventories()->where('status', 'publish')->get()->map(fn($inventory) => [
                ...[
                    'id' => $inventory->id,
                    'uuid' => $inventory->uuid,
                    'store' => $inventory->store ? [
                        'id' => $inventory->store->id,
                        'uuid' => $inventory->store->uuid,
                        'user_id' => $inventory->store->user_id,
                        'name' => $inventory->store->name,
                        'logo' => $inventory->store->get_logo,
                        'type' => $inventory->store->type,
                        'direct_post' => $inventory->store->direct_post,
                        'meta' => $inventory->store->get_meta,
                        'status' => $inventory->store->status,
                        'created_at' => $inventory->store->created_at,
                        'updated_at' => $inventory->store->updated_at,
                    ] : null,
                    'storage' => $inventory->storage ? [
                        'id' => $inventory->storage->id,
                        'uuid' => $inventory->storage->uuid,
                        'name' => $inventory->storage->name,
                        'address' => $inventory->storage->address,
                        'product_count' => $inventory->storage->product_count,
                        'created_at' => $inventory->storage->created_at,
                        'updated_at' => $inventory->storage->updated_at,
                    ] : null,
                    'props' => $inventory->raw_props,
                    'price' => $inventory->price,
                    'discount_price' => $inventory->discount_price,
                    'discount_expire' => $inventory->discount_expire,
                    'discount_percent' => $inventory->get_discount_percent,
                    'count' => $inventory->count,
                    'min_sale' => $inventory->min_sale,
                    'max_sale' => $inventory->max_sale,
                    'price_changes' => $inventory->get_price_changes_chart,
                    'status' => $inventory->status,
                    'created_at' => $inventory->created_at,
                    'updated_at' => $inventory->updated_at,
                ]
            ]),
            'commission' => $product->commission,
            'is_vip' => $product->is_vip,
            'comment_status' => $product->comment_status,
            'question_status' => $product->question_status,
            'status' => $product->status,
            'created_at' => $product->created_at,
            'updated_at' => $product->updated_at,
        ]);

        return response()->json([
            'success' => true,
            'response' => $products,
            'code' => 200,
        ]);
    }
}
