<?php

namespace App\Http\Controllers\Main\Brand;

use App\Http\Controllers\Controller;
use App\Models\Brand;
use App\Services\Main\Brands\BrandsService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Inertia\Inertia;

class BrandsController extends Controller
{
    public function show(Request $request, Brand $brand, BrandsService $brandsService)
    {
        // check product status
        if ($brand->status != 'publish')
            abort(404);

        // increase view count
        $brand->update([
            'view_count' => $brand->view_count + 1
        ]);

        $seo = $brandsService->seo($brand);

        // brand custom product list
        $products['latest-products'] = $brand->products()->where('status', 'publish')->orderBy('created_at', 'desc')->take(8);
        $products['popular-products'] = $brand->products()->where('status', 'publish')->withCount(['productComments as comment_rating_average' => function ($query) {
            $query->select(DB::raw('avg(rating)'));
        }])->orderBy('comment_rating_average', 'desc')->limit(12);
        $products['discounted-products'] = $brand->products()->where('status', 'publish')->whereHas('inventories', function ($query) {
            $query->where('status', 'publish')->where('count', '>', 0)->where('discount_price', '!=', null);
        })->inRandomOrder()->limit(12);

        foreach ($products as $key => $value) {
            $products[$key] = $products[$key]->get()->map(fn($product) => [
                'id' => $product->id,
                'title' => $product->title,
                'slug' => $product->slug,
                'get_images' => $product->get_images,
                'comments_info' => $product->comments_info,
                'in_stock_status' => $product->in_stock_status,
                'inventory' => $product->inventory,
                'best_normal_price' => $product->best_normal_price,
                'unique_color' => $product->unique_color,
                'best_price' => $product->best_price,
                'is_vip' => $product->is_vip,
            ]);
        }

        // brand article list
        $latestArticles = $brand->articles()->where('status', 'publish')->orderBy('created_at', 'desc')->take(6)->get()->map(fn($article) => [
            'id' => $article->id,
            'title' => $article->title,
            'slug' => $article->slug,
            'get_image' => $article->get_image,
            'jalali_created_at' => $article->jalali_created_at,
            'user' => [
                'id' => $article->user->id,
                'get_name' => $article->user->get_name,
                'get_avatar' => $article->user->get_avatar,
                'username' => $article->user->username,
                'seller' => $article->user->seller,
                'store' => $article->user->store != null ? [
                    'id' => $article->user->store->id,
                    'get_logo' => $article->user->store->get_logo,
                    'name' => $article->user->store->name,
                    'uuid' => $article->user->store->uuid,
                ] : null,

            ],
            'category' => [
                'title' => $article->articleCategory->title,
                'slug' => $article->articleCategory->slug,
            ],
        ]);

        $view = Inertia::render('Main/Brand/Show', [
            'seo' => $seo,
            'brand' => fn() => [
                'id' => $brand->id,
                'title' => $brand->title,
                'title_en' => $brand->title_en,
                'slug' => $brand->slug,
                'content' => $brand->content,
                'get_logo' => $brand->get_logo,
                'get_banner' => $brand->get_banner,
                'website' => $brand->website,
                'products_count' => $brand->products_count,
                'get_categories' => $brand->get_categories->map(fn($catLevel1) => [
                    'id' => $catLevel1['safe']['id'],
                    'title' => $catLevel1['safe']['title'],
                    'slug' => $catLevel1['safe']['slug']
                ]),
            ],
            'latest_products' => fn() => $products['latest-products'],
            'popular_products' => fn() => $products['popular-products'],
            'discounted_products' => fn() => $products['discounted-products'],
            'latest_articles' => fn() => $latestArticles,
        ]);

        $view->title($seo['title'])
            ->image($brand->get_logo)
            ->tag('<link rel="canonical" href="' . str_replace('%', '%%', $seo['canonical'] !== null ? $seo['canonical'] : urldecode(url()->full())) . '" />');

        if ($seo['description'] != null) {
            $view->description($seo['description']);
        }

        // og and twitter card
        $view->ogMeta()
            ->ogUrl(urldecode(url()->full()))
            ->ogLocale('fa_IR')
            ->tag('<meta property="og:site_name" content="' . $seo['site_name'] . '" />')
            ->twitterSummaryCard()
            ->twitterSite($request->url())
            ->twitterCreator($seo['site_name'])
            ->tag('<script type="application/ld+json">' . str_replace('%', '%%', $seo['schema']) . '</script>'); // schema

        if ($seo['type'] != null) {
            $view->ogType($seo['type']);
        }

        return $view;
    }
}
