<?php

namespace App\Http\Controllers\Main\Forms;

use App\Helpers\CaptchaHelper;
use App\Http\Controllers\Controller;
use App\Models\Form;
use App\Models\Setting;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Cache;
use Inertia\Inertia;

class FormsController extends Controller
{
    use CaptchaHelper;

    /** seo meta data */
    public array $seo = [
        'site_name' => null,
        'title' => null,
        'description' => null,
        'keywords' => null,
        'canonical' => null,
    ];

    /**
     * Display the specified product
     */
    public function show(Request $request, Form $form)
    {
        // increase view count
        $form->update([
            'view_count' => $form->view_count + 1
        ]);

        $shopTitle = Cache::remember('setting_shop_title', now()->addMonth(), function () {
            return Setting::where('key', 'shop_title')->first() != null ? Setting::where('key', 'shop_title')->first()->value : null;
        });

        // main
        $this->seo['site_name'] = $shopTitle;
        $this->seo['title'] = ($form->seo_title != null ? $form->seo_title : $form->title) . ' - ' . $shopTitle;
        $this->seo['description'] = $form->seo_description;
        $this->seo['keywords'] = $form->get_seo_keywords;
        $this->seo['canonical'] = $form->seo_canonical;

        $view = Inertia::render('Main/Forms/Show', [
            'form' => fn() => [
                'id' => $form->id,
                'title' => $form->title,
                'slug' => $form->slug,
                'description' => $form->description,
                'get_fields' => $form->get_fields,
            ],
            'seo' => fn() => $this->seo,
        ]);

        $view->title($this->seo['title']);
        if ($this->seo['description'] != null) {
            $view->description($this->seo['description']);
        }
        $view->tag('<link rel="canonical" href="' . str_replace('%', '%%', $this->seo['canonical'] !== null ? $this->seo['canonical'] : urldecode(url()->full())) . '" />')
            ->ogMeta()
            ->ogUrl(urldecode(url()->full()))
            ->ogLocale('fa_IR')
            ->tag('<meta property="og:site_name" content="' . $this->seo['site_name'] . '" />')
            ->twitterSummaryCard()
            ->twitterSite($request->url())
            ->twitterCreator($this->seo['site_name']);

        return $view;
    }

    /**
     * Display the specified product
     */
    public function answer(Request $request, Form $form)
    {
        $request->validate([
            'captcha_args' => ['required', 'array'],
            'captcha' => 'required',
            'answer' => 'required',
        ]);

        // check for type
        foreach ($request->answer as $field) {
            if ($field['status'] == 'necessary' && $field['type'] != 'checkbox')
                if (!isset($field['value']) || $field['value'] == null || $field['value'] == '')
                    return redirect()->back()->withErrors([__('messages.please_fill_all_neccesary_fields')]);
        }

        // validate captcha
        if (!$this->validateCaptcha($request->captcha_args, $request->captcha))
            return redirect()->back()->withErrors([__('messages.captcha_is_wrong')]);

        /** @var \App\Models\User $user */
        $user = auth()->user();

        $form->formAnswers()->create([
            'answer' => serialize($request->answer),
            'user_id' => $user != null ? $user->id : null,
            'flag' => 'unread'
        ]);

        return redirect()->back()->with('message', [__('messages.form_answer_sent')]);
    }
}
