<?php

namespace App\Http\Controllers\Main\Products;

use App\Http\Controllers\Controller;
use App\Models\Compare;
use App\Models\Product;
use App\Models\Setting;
use App\Services\Main\Products\CompareService;
use Illuminate\Http\Request;
use Inertia\Inertia;

class CompareController extends Controller
{
    /**
     * show product compare page
     */
    public function show(Request $request, CompareService $compareService)
    {
        /** @var \App\Models\User $user */
        $user = auth()->user();

        // validate compares category
        $compareService->validateComparesCategory($compareService->getComparesProducts($user));

        return Inertia::render('Main/Products/Compare', [
            'products' => fn() => $compareService->getSearchProducts($request, $compareService->getComparesProducts($user)),
            'compares' => fn() => $compareService->getComparesProducts($user),
        ])->title(__('messages.sentence.compare_products'));
    }

    /**
     * add product to compare page
     */
    public function add(Request $request, Product $product, CompareService $compareService)
    {
        /** @var \App\Models\User $user */
        $user = auth()->user();

        if ($request->type == 'start') {
            $user->compares()->whereIn('id', $user->compares()->get()->pluck('id')->toArray())->delete();
            $user->compares()->create([
                'product_id' => $product->id
            ]);
        } else {
            // check unique
            if ($user->compares()->where('product_id', $product->id)->first() != null) {
                return redirect()->route('main.compare.show');
            }

            // check count
            if ($user->compares()->count() >= 4) {
                return redirect()->route('main.compare.show')->withErrors([__('messages.can_not_add_more_than_four_product')]);
            }

            // add new
            if ($compareService->isComparesCategorySame($compareService->getComparesProducts($user), $product)) {
                $user->compares()->create([
                    'product_id' => $product->id
                ]);
            } else {
                return redirect()->route('main.compare.show')->withErrors([__('messages.compare_category_not_same')]);
            }
        }

        return redirect()->route('main.compare.show')->with('message', [__('messages.product_added_to_compare')]);
    }

    /**
     * add product to compare page
     */
    public function delete(Compare $compare)
    {
        /** @var \App\Models\User $user */
        $user = auth()->user();

        if ($compare->user_id == $user->id) {
            $compare->delete();
        }

        return redirect()->route('main.compare.show')->with('message', [__('messages.product_deleted_from_compare')]);
    }
}