<?php

namespace App\Http\Controllers\Main\Sitemap;

use App\Http\Controllers\Controller;
use App\Models\Article;
use App\Models\ArticleCategory;
use App\Models\ArticleTag;
use App\Models\Brand;
use App\Models\Form;
use App\Models\Page;
use App\Models\Product;
use App\Models\ProductCategory;
use App\Models\ProductTag;
use App\Models\Store;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Spatie\Sitemap\Sitemap;
use Spatie\Sitemap\SitemapIndex;
use Spatie\Sitemap\Tags\Url;

class SitemapController extends Controller
{
    public function index(Request $request)
    {
        $sitemaps = collect([
            "statics",
            "product_categories",
            "product_tags",
            "article_categories",
            "article_tags",
            "pages",
            "forms",
            "stores",
            "brands",
        ]);

        $masterSitemap = SitemapIndex::create();

        $productCount = Product::where('status', 'publish')->count();
        $productPages = ceil($productCount / 100);
        for ($i = 1; $i <= $productPages; $i++) {
            $masterSitemap->add(asset("/sitemap/products-$i.xml"));
        }

        $articleCount = Article::where('status', 'publish')->count();
        $articlePages = ceil($articleCount / 100);
        for ($i = 1; $i <= $articlePages; $i++) {
            $masterSitemap->add(asset("/sitemap/articles-$i.xml"));
        }

        $sitemaps->each(function ($title) use ($masterSitemap) {
            $masterSitemap->add(asset("/sitemap/{$title}.xml"));
        });

        return $masterSitemap->toResponse($request)->header('Content-Type', 'application/xml');
    }

    /**
     * sitemap urls
     */
    public function urls(Request $request, $title)
    {
        $sitemap = Sitemap::create();

        if (preg_match('/^products-(\d+)$/', $title, $matches)) {
            $page = (int)$matches[1];
            $products = Product::where('status', 'publish')
                ->skip(($page - 1) * 100)
                ->take(100)
                ->get();

            foreach ($products as $product) {
                $sitemap->add(
                    Url::create(urldecode(route('main.products.show', $product->slug)))
                        ->setPriority(0.8)
                        ->setChangeFrequency(Url::CHANGE_FREQUENCY_DAILY)
                        ->setLastModificationDate(Carbon::create($product->updated_at))
                );
            }

            return $sitemap->toResponse($request)->header('Content-Type', 'application/xml');
        }

        if (preg_match('/^articles-(\d+)$/', $title, $matches)) {
            $page = (int)$matches[1];
            $articles = Article::where('status', 'publish')
                ->skip(($page - 1) * 100)
                ->take(100)
                ->get();

            foreach ($articles as $article) {
                $sitemap->add(
                    Url::create(urldecode(route('blog.articles.show', $article->slug)))
                        ->setPriority(0.8)
                        ->setChangeFrequency(Url::CHANGE_FREQUENCY_DAILY)
                        ->setLastModificationDate(Carbon::create($article->updated_at))
                );
            }

            return $sitemap->toResponse($request)->header('Content-Type', 'application/xml');
        }

        switch ($title) {
            case "statics":
                $sitemap->add(Url::create(route('index')));
                $sitemap->add(Url::create(route('blog.index')));
                $sitemap->add(Url::create(route('main.faq.index')));
                break;

            case "product_categories":
                ProductCategory::where('status', 'publish')->get()->each(function (ProductCategory $cat) use ($sitemap) {
                    $sitemap->add(
                        Url::create(route('main.products.index', ['cat' => $cat->slug]))
                            ->setLastModificationDate(Carbon::create($cat->updated_at))
                    );
                });
                break;

            case "product_tags":
                ProductTag::where('status', 'publish')->get()->each(function (ProductTag $tag) use ($sitemap) {
                    $sitemap->add(
                        Url::create(route('main.products.index', ['tag' => $tag->slug]))
                            ->setLastModificationDate(Carbon::create($tag->updated_at))
                    );
                });
                break;

            case "article_categories":
                ArticleCategory::where('status', 'publish')->get()->each(function (ArticleCategory $cat) use ($sitemap) {
                    $sitemap->add(
                        Url::create(route('blog.articles.index', ['cat' => $cat->slug]))
                            ->setLastModificationDate(Carbon::create($cat->updated_at))
                    );
                });
                break;

            case "article_tags":
                ArticleTag::where('status', 'publish')->get()->each(function (ArticleTag $tag) use ($sitemap) {
                    $sitemap->add(
                        Url::create(route('blog.articles.index', ['tag' => $tag->slug]))
                            ->setLastModificationDate(Carbon::create($tag->updated_at))
                    );
                });
                break;

            case "pages":
                Page::where('status', 'publish')->get()->each(function (Page $page) use ($sitemap) {
                    $sitemap->add(
                        Url::create(route('main.pages.show', $page->slug))
                            ->setLastModificationDate(Carbon::create($page->updated_at))
                    );
                });
                break;

            case "forms":
                Form::where('status', 'publish')->get()->each(function (Form $form) use ($sitemap) {
                    $sitemap->add(
                        Url::create(route('main.forms.show', $form->slug))
                            ->setLastModificationDate(Carbon::create($form->updated_at))
                    );
                });
                break;

            case "stores":
                Store::where('status', 'active')->get()->each(function (Store $store) use ($sitemap) {
                    $sitemap->add(
                        Url::create(route('main.stores.show', $store->uuid))
                            ->setLastModificationDate(Carbon::create($store->updated_at))
                    );
                });
                break;

            case "brands":
                Brand::where('status', 'publish')->get()->each(function (Brand $brand) use ($sitemap) {
                    $sitemap->add(
                        Url::create(route('main.brands.show', $brand->slug))
                            ->setLastModificationDate(Carbon::create($brand->updated_at))
                    );
                });
                break;

            default:
                abort(404);
        }

        return $sitemap->toResponse($request)->header('Content-Type', 'application/xml');
    }
}
