<?php

namespace App\Http\Controllers\Main\Stores;

use App\Http\Controllers\Controller;
use App\Models\Color;
use App\Models\Store;
use App\Services\Main\Products\ProductsArchiveService;
use Illuminate\Http\Request;
use Inertia\Inertia;

class StoresController extends Controller
{
    /**
     * show store
     */
    public function show(Request $request, Store $store, ProductsArchiveService $productsArchiveService)
    {
        if ($store->status != 'active') {
            abort(403);
        }

        // get products
        $products = $productsArchiveService->getProducts($request, $store);
        $seo = $products['seo'];

        $view = Inertia::render('Main/Stores/Show', [
            'seo' => fn() => $seo,
            'store' => fn() => [
                'id' => $store->id,
                'uuid' => $store->uuid,
                'name' => $store->name,
                'get_logo' => $store->get_logo,
                'jalali_created_at' => $store->jalali_created_at,
                'get_meta' => $store->get_meta,
                'bio' => $store->bio
            ],
            'products' => fn() => $products['products'],
            'filters' => fn() => [
                'brands' => $products['brands'],
                'colors' => Color::all(),
                'price' => $products['price'],
            ],
            'keywords' => fn() => $products['keywords'],
        ]);

        $view->title($seo['title'] . ' - ' . $store->name)
            ->image($store->get_logo)
            ->tag('<meta name="author" content="' . str_replace('%', '%%', $store->name) . '" />')
            ->tag('<link rel="canonical" href="' . urldecode(url()->full()) . '" />');

        if ($seo['description'] != null) {
            $view->description($seo['description']);
        }

        // og and twitter card
        $view->ogMeta()
            ->ogUrl(urldecode(url()->full()))
            ->ogLocale('fa_IR')
            ->tag('<meta property="og:site_name" content="' . $seo['site_name'] . '" />')
            ->twitterSummaryCard()
            ->twitterSite($request->url())
            ->twitterCreator($seo['site_name'])
            ->tag('<script type="application/ld+json">' . str_replace('%', '%%', $seo['schema']) . '</script>'); // schema

        if ($seo['type'] != null) {
            $view->ogType($seo['type']);
        }

        return $view;
    }
}
