<?php

namespace App\Http\Controllers\Store\Account;

use App\Helpers\DropdownListHelper;
use App\Http\Controllers\Controller;
use App\Models\Token;
use App\Models\User;
use App\Rules\NationalNumber;
use App\Rules\Username;
use Exception;
use Illuminate\Support\Facades\Session;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Validation\Rule;
use Inertia\Inertia;
use Inertia\Response;
use Morilog\Jalali\Jalalian;

class ProfileController extends Controller
{
    use DropdownListHelper;

    /**
     * edit profile
     */
    public function edit(Request $request): Response
    {
        /** @var App\Models\User $user */
        $user = auth()->user();

        return Inertia::render('Store/Account/Edit', [
            'user' => [
                'id' => $user->id,
                'name' => $user->name,
                'get_name' => $user->get_name,
                'username' => $user->username,
                'phone' => $user->phone,
                'email' => $user->email,
                'email_verified_at' => $user->email_verified_at,
                'avatar' => $user->avatar,
                'get_avatar' => $user->get_avatar,
                'national_number' => $user->national_number,
                'birthday' => $user->birthday,
                'jalali_birthday' => $user->jalali_birthday,
                'gender' => $user->gender,
                'bio' => $user->bio,
                'get_social' => $user->get_social,
                'seller' => $user->seller,
                'status' => $user->status,
                'jalali_created_at' => $user->jalali_created_at,
                'jalali_updated_at' => $user->jalali_updated_at,
            ],
            'genderList' => $this->genderList()
        ]);
    }

    /**
     * update profile
     */
    public function update(Request $request): RedirectResponse
    {
        /** validation */
        $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'username' => ['required', 'string', 'regex:/\w*$/', 'min:4', 'max:20', 'unique:users,username,' . auth()->user()->id, new Username()]
        ]);
        if ($request->exists('email') && $request->email != '') {
            $request->validate([
                'email' => ['required', 'string', 'lowercase', 'email', 'max:255', Rule::unique(User::class)->ignore(auth()->user()->id)],
            ]);
        }
        if ($request->exists('national_number') && $request->national_number != '') {
            $request->validate([
                'national_number' => ['required', 'string', new NationalNumber()],
            ]);
        }

        /** birthday */
        $birthday = null;
        if ($request->filled('birthday')) {
            try {
                $jalali = Jalalian::fromFormat('Y/m/d', $request->birthday);
                $timestamp = $jalali->getTimestamp();
                if ($timestamp < 0) {
                    $birthday = null;
                } else {
                    $carbon = $jalali->toCarbon();
                    if ($carbon->isFuture()) {
                        $birthday = null;
                    } else {
                        $birthday = $carbon->setTimezone('Asia/Tehran')->format('Y-m-d H:i:s');
                    }
                }
            } catch (\Exception $e) {
                $birthday = null;
            }
        }

        /** gender */
        $gender = null;
        if ($request->exists('gender') && $request->gender != '') {
            if ($request->gender == __('messages.man') || $request->gender == __('messages.woman') || $request->gender == __('messages.unknown')) {
                $gender = $request->gender;
            }
        }

        /** social */
        $social = null;
        if ($request->exists('social') && $request->social != '') {
            if (is_array($request->social) && array_key_exists('instagram', $request->social) && array_key_exists('telegram', $request->social) && array_key_exists('twitter', $request->social) && array_key_exists('facebook', $request->social)) {
                $social = serialize($request->social);
            }
        }

        /** @var App\Models\User $user */
        $user = auth()->user();

        $user->update([
            'name' => $request->name,
            'username' => $request->username,
            'email' => $user->email_verified_at == null ? $request->email : $user->email,
            'avatar' => $request->exists('avatar') && $request->avatar != null ? $request->avatar : null,
            'national_number' => $request->exists('national_number') ? $request->national_number : null,
            'birthday' => $birthday,
            'gender' => $gender,
            'bio' => $request->exists('bio') ? $request->bio : null,
            'social' => $social,
        ]);

        return redirect()->back()->with('message', [__('messages.profile_updated')]);
    }

    /**
     * active sessions list
     */
    public function sessions(): Response
    {
        /** @var App\Models\User $user */
        $user = auth()->user();
        $sessions = Token::where('user_id', $user->id)->where('used', true)->select('id', 'user_id', 'ip', 'meta', 'created_at')->orderBy('created_at', 'desc')->paginate(20);

        foreach ($sessions as $session) {
            $deviceMeta = unserialize($session->meta);
            if ($deviceMeta['location'] == null) {
                try {
                    $location = $this->ipDetails($session->ip);
                    $deviceMeta['location'] = [
                        'country' => $location->country,
                        'city' => $location->city,
                        'latitude' => $location->country,
                        'longitude' => $location->country
                    ];
                    $session->update([
                        'meta' => serialize($deviceMeta)
                    ]);
                } catch (Exception $e) {
                    $deviceMeta['location'] = null;
                }
            }
        }

        $currentSesison = $user->tokens()->where('session', session()->getId())->first();

        return Inertia::render('Store/Account/Sessions', [
            'sessions' => $sessions,
            'current_session' =>  $currentSesison != null ?  $currentSesison->id : null
        ]);
    }

    /**
     * terminate user session
     * @method delete
     */
    public function terminate(Request $request): RedirectResponse
    {
        $request->validate([
            'session_id' => ['required']
        ]);

        $user = auth()->user();

        $session = Token::where('id', $request->session_id)->where('user_id', $user->id)->first();

        if ($session != null) {

            if ($session->session == session()->getId()) {
                Auth::logout();

                $request->session()->invalidate();

                $request->session()->regenerateToken();

                $session->delete();

                return redirect()->route('index')->with('message', [__('messages.session_terminated')]);
            }

            $result = Session::getHandler()->destroy($session->session);

            if ($result) {
                $session->delete();
                return redirect()->back()->with('message', [__('messages.session_terminated')]);
            }
            return redirect()->back()->withErrors(['error' => __('messages.unknown_error')]);
        }

        return redirect()->back()->withErrors([__('messages.invalid_session')]);
    }
}
