<?php

namespace App\Http\Controllers\Store\Products;

use App\Models\Product;
use App\Http\Controllers\Controller;
use App\Models\Brand;
use App\Models\ProductCategory;
use App\Models\ProductTag;
use App\Models\Specification;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Inertia\Inertia;

class ProductsController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        /** @var App\Models\User $user */
        $user = auth()->user();
        $store = $user->store;

        if ($store->direct_post && !$store->transportation_status) {
            return redirect()->route('seller.transportation.index')->withErrors([__('messages.post_ways_required')]);
        }

        /** seller products list */
        /** @var LengthAwarePaginator */
        $products = $store->products()
            ->where('status', '!=', 'deleted')
            ->where('status', '!=', 'deep_deleted')
            ->withCount(['consignmentItems' => function ($query) {
                $query->where(function ($query) {
                    $query->where('status', 'sent')->orWhere('status', 'delivered');
                })->select(DB::raw('SUM(count)'));
            }])
            ->when($request->input('search'), function ($query, $search) {
                $query->where(DB::raw('concat(`title`, `uuid`)'), 'like', "%{$search}%");
            })->when($request->input('createdByYou'), function ($query, $createdByYou) use ($store) {
                if ($createdByYou) {
                    $query->where('user_id', $store->user->id);
                }
            })->when($request->input('hasPrice'), function ($query, $hasPrice) use ($store) {
                if ($hasPrice) {
                    $query->whereHas('inventories', function ($query) use ($store) {
                        $query->where('store_id', $store->id);
                    });
                }
            })->when($request->input('mostSale'), function ($query, $mostSale) {
                if ($mostSale) {
                    $query->orderBy('consignment_items_count', 'desc');
                }
            })->when($request->input('mostViewer'), function ($query, $mostViewer) {
                if ($mostViewer) {
                    $query->orderBy('view_count', 'desc');
                }
            })
            ->orderByRaw("FIELD(status , 'awaiting') DESC")
            ->orderBy('product_store.created_at', 'DESC')
            ->paginate(20)->through(fn($product) => [
                'id' => $product->id,
                'user_id' => $product->user_id,
                'store_id' => $product->store_id,
                'uuid' => $product->uuid,
                'title' => $product->title,
                'slug' => $product->slug,
                'get_images' => $product->get_images,
                'in_stock_status' => $product->in_stock_status,
                'inventory' => $product->inventory,
                'best_price' => $product->best_price,
                'unique_color' => $product->unique_color,
                'get_formatted_view_count' => $product->get_formatted_view_count,
                'is_vip' => $product->is_vip,
                'reason' => $product->reason,
                'status' => $product->status,
                'jalali_created_at' => $product->jalali_created_at,
                'jalali_updated_at' => $product->jalali_updated_at,
                'user' => [
                    'id' => $product->user->id,
                    'get_name' => $product->user->get_name,
                    'type' => $product->user->type,
                    'seller' => $product->user->seller,
                    'store' => $product->user->store != null ? [
                        'id' => $product->user->store->id,
                        'name' => $product->user->store->name,
                        'uuid' => $product->user->store->uuid,
                    ] : null
                ],
                'store_has_inventory' => $product->inventories()->where('store_id', $store->id)->count() > 0 ? true : false
            ]);

        $products->withQueryString();

        /** searchableProducts */
        /** @var LengthAwarePaginator */
        $searchableProducts = Product::query()->where('status', 'publish')
            ->when($request->input('s'), function ($query, $search) {
                $query->where(DB::raw('concat(`title`, `uuid`)'), 'like', "%{$search}%");
            })->orderBy('created_at', 'desc')->paginate(8, ['*'], 'p')->through(fn($product) => [
                'id' => $product->id,
                'uuid' => $product->uuid,
                'title' => $product->title,
                'slug' => $product->slug,
                'get_images' => $product->get_images,
                'in_stock_status' => $product->in_stock_status,
                'inventory' => $product->inventory,
                'best_price' => $product->best_price,
                'unique_color' => $product->unique_color,
                'status' => $product->status,
                'jalali_created_at' => $product->jalali_created_at,
                'jalali_updated_at' => $product->jalali_updated_at,
                'has_in_store_list' => $product->stores()->where('id', $store->id)->count() > 0 ? true : false,
            ]);
        $searchableProducts->withQueryString();

        return Inertia::render('Store/Products/Products/List', [
            'products' => $products,
            'searchableProducts' => $searchableProducts,
        ]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $categoriesWithChilds = null;
        $categories = ProductCategory::where('status', 'publish')->where('parent_id', 0)->select('id', 'title', 'parent_id')->get()->map(fn($category) => [
            'id' => $category->id,
            'title' => $category->title,
            'slug' => $category->slug,
            'specifications' => $category->specifications()->get()->map(fn($specification) => [
                'id' => $specification->id,
                'title' => $specification->title,
                'keys' => $specification->specificationKeys()->get()->map(fn($specificationKey) => [
                    'id' => $specificationKey->id,
                    'key' => $specificationKey->key,
                    'value' => $specificationKey->default_value,
                ])->toArray()
            ])->toArray(),
            'childs' => $category->childs,
            'level' => 'one',
        ])->toArray();

        foreach ($categories as $category) {
            $categoriesWithChilds[] = $category;
            foreach ($category['childs'] as $childLevelTwo) {
                $categoriesWithChilds[] = $childLevelTwo;
                foreach ($childLevelTwo['childs'] as $childLevelThree) {
                    $categoriesWithChilds[] = $childLevelThree;
                }
            }
        }

        return Inertia::render('Store/Products/Products/Create', [
            'categories' => $categoriesWithChilds,
        ]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'title' => 'required',
            'category' => 'required',
            'price_model' => 'required',
            'question_status' => 'required',
            'comment_status' => 'required',
        ]);

        /** @var App\Models\User $user */
        $user = auth()->user();
        $store = $user->store;

        /** brand */
        $brand = null;
        if ($request->exists('brand') && $request->brand != null && $request->brand != '') {
            $findBrand = Brand::where('title', $request->brand)->get()->first();
            if ($findBrand != null) {
                $brand = $findBrand->id;
            } else {
                $brand = $user->brands()->create([
                    'uuid' => 'brand-' . rand(1000000, 9999999),
                    'title' => $request->brand,
                    'status' => 'publish',
                ]);
                $brand = $brand->id;
            }
        }

        /** validate duplicated slug */
        if ($request->exists('slug') && $request->slug != null) {
            $productWithSlug = Product::where('slug', $request->slug)->first();
            if ($productWithSlug != null) {
                return redirect()->back()->withErrors([__('messages.slug_is_exist')]);
            }
        }

        $product = $user->products()->create([
            'edited_by' => $user->id,
            'brand_id' => $brand,
            'category_id' => $request->category['id'],
            'uuid' => 'p-' . rand(1000000, 9999999),
            'title' => $request->title,
            'title_en' => $request->exists('title_en') ? $request->title_en : null,
            'slug' => $request->exists('slug') && $request->slug != null ? $request->slug : null,
            'content' => $request->exists('content') ? $request->content : null,
            'warnings' => $request->exists('warnings') ? $request->warnings : null,

            'images' => $request->exists('images') ? serialize($request->images) : null,

            'attrs' => $request->exists('attrs') ? serialize($request->attrs) : null,

            'seo_title' => $request->exists('seo_title') ? $request->seo_title : null,
            'seo_description' => $request->exists('seo_description') ? $request->seo_description : null,
            'seo_keywords' => $request->exists('seo_keywords') ? $request->seo_keywords : null,
            'seo_canonical' => $request->exists('seo_canonical') ? $request->seo_canonical : null,

            'in_stock_status' => true,
            'inquiry_options' => serialize($request->inquiry_options),

            'commission' => 0,

            'is_vip' => false,
            'comment_status' => $request->comment_status,
            'question_status' => $request->question_status,

            'reason' => 'create',
            'status' => 'awaiting',
        ]);

        /** add price model */
        if ($request->exists('price_model') && is_array($request->price_model)) {
            $pmID = 0;
            foreach ($request->price_model as $pm) {
                if ($pm['active'] && $pm['name'] != null) {
                    $pmModel = $product->productProps()->create([
                        'child' => $pmID,
                        'name' => $pm['name'],
                        'type' => $pm['type'],
                    ]);

                    $pmID = $pmModel->id;
                }
            }
        }

        try {
            /** add specifications */
            if ($request->exists('specifications') && is_array($request->specifications)) {
                foreach ($request->specifications as $specification) {
                    foreach ($specification['keys'] as $key) {
                        if ($key['value'] != null) {
                            $product->specificationValues()->create([
                                'specification_key_id' => $key['id'],
                                'value' => $key['value'],
                            ]);
                        }
                    }
                }
            }
        } catch (Exception $e) {
        }


        /** create and sync tags */
        if ($request->exists('tags') && $request->tags != null && $request->tags != '') {
            $tags = [];
            foreach (json_decode($request->tags) as $tag) {
                $findTag = ProductTag::where('title', $tag->value)->get()->first();
                if ($findTag != null) {
                    $tags[] = $findTag->id;
                } else {
                    $createdTag = ProductTag::create([
                        'title' => $tag->value,
                        'status' => 'publish',
                    ]);
                    $tags[] = $createdTag->id;
                }
            }
            $product->productTags()->sync($tags);
        }

        /** add product to seller list */
        $store->products()->attach($product->id);

        /** Report Action */
        $reportMessage = __("messages.report_product_created");
        $this->report($reportMessage, 'product', $product);
        /******************/

        return redirect()->back()->with('message', [__('messages.product_created')]);
    }

    /**
     * Display the specified resource.
     */
    public function show(Product $product)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Product $product)
    {
        /** @var App\Models\User $user */
        $user = auth()->user();
        $store = $user->store;

        if ($product->status == 'publish' || $product->status == 'draft') {
            return redirect()->route('seller.products.index')->withErrors([__('messages.unauthorized')]);
        }

        if ($product->user_id != $user->id) {
            return redirect()->route('seller.products.index')->withErrors([__('messages.unauthorized')]);
        }

        $categoriesWithChilds = null;
        $categories = ProductCategory::where('status', 'publish')->where('parent_id', 0)->get()->map(fn($category) => [
            'id' => $category->id,
            'title' => $category->title,
            'slug' => $category->slug,
            'specifications' => $category->specifications()->get()->map(fn($specification) => [
                'id' => $specification->id,
                'title' => $specification->title,
                'keys' => $specification->specificationKeys()->get()->map(fn($specificationKey) => [
                    'id' => $specificationKey->id,
                    'value_id' => $specificationKey->specificationValues()->where('product_id', $product->id)->first() != null ? $specificationKey->specificationValues()->where('product_id', $product->id)->first()->id : null,
                    'key' => $specificationKey->key,
                    'value' => $specificationKey->specificationValues()->where('product_id', $product->id)->first() != null ? $specificationKey->specificationValues()->where('product_id', $product->id)->first()->value : $specificationKey->default_value,
                ])->toArray()
            ])->toArray(),
            'childs' => $category->childs,
            'level' => 'one',
        ])->toArray();

        foreach ($categories as $category) {
            $categoriesWithChilds[] = $category;
            foreach ($category['childs'] as $childLevelTwo) {
                $categoriesWithChilds[] = $childLevelTwo;
                foreach ($childLevelTwo['childs'] as $childLevelThree) {
                    $categoriesWithChilds[] = $childLevelThree;
                }
            }
        }

        return Inertia::render('Store/Products/Products/Edit', [
            'product' => [
                'id' => $product->id,
                'uuid' => $product->uuid,
                'title' => $product->title,
                'title_en' => $product->title_en,
                'slug' => $product->slug,
                'get_images_id' => $product->get_images_id,
                'get_images_link' => $product->get_images_link,
                'get_attrs' => $product->get_attrs,
                'price_model' => $product->price_model,
                'get_tags_string' => $product->get_tags_string,
                'content' => $product->content,
                'warnings' => $product->warnings,
                'seo_title' => $product->seo_title,
                'seo_description' => $product->seo_description,
                'seo_keywords' => $product->seo_keywords,
                'seo_canonical' => $product->seo_canonical,
                'in_stock_status' => $product->in_stock_status,
                'get_inquiry_options' => $product->get_inquiry_options,
                'question_status' => $product->question_status,
                'comment_status' => $product->comment_status,
                'reason' => $product->reason,
                'status' => $product->status,

                'brand' => $product->brand != null ? [
                    'id' => $product->brand->id,
                    'title' => $product->brand->title,
                ] : null,
                'category' => [
                    'id' => $product->productCategory->id,
                ],
            ],
            'categories' => $categoriesWithChilds,
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Product $product)
    {
        $request->validate([
            'title' => 'required',
            'category' => 'required',
            'question_status' => 'required',
            'comment_status' => 'required',
            'status' => 'required',
        ]);

        /** @var App\Models\User $user */
        $user = auth()->user();
        $store = $user->store;

        if ($product->status == 'publish' || $product->status == 'draft') {
            return redirect()->route('seller.products.index')->withErrors([__('messages.unauthorized')]);
        }

        if ($product->user_id != $user->id) {
            return redirect()->route('seller.products.index')->withErrors([__('messages.unauthorized')]);
        }

        /** brand */
        $brand = null;
        if ($request->exists('brand') && $request->brand != null && $request->brand != '') {
            $findBrand = Brand::where('title', $request->brand)->get()->first();
            if ($findBrand != null) {
                $brand = $findBrand->id;
            } else {
                $brand = $user->brands()->create([
                    'uuid' => 'brand-' . rand(1000000, 9999999),
                    'title' => $request->brand,
                    'status' => 'publish',
                ]);
                $brand = $brand->id;
            }
        }

        /** validate duplicated slug */
        if ($request->exists('slug') && $request->slug != null && $request->slug != $product->slug) {
            $productWithSlug = Product::where('slug', $request->slug)->first();
            if ($productWithSlug != null) {
                return redirect()->back()->withErrors([__('messages.slug_is_exist')]);
            }
        }

        $product->update([
            'edited_by' => $user->id,
            'brand_id' => $brand,
            'category_id' => $request->category['id'],
            'title' => $request->title,
            'title_en' => $request->exists('title_en') ? $request->title_en : $product->title_en,
            'slug' => $request->exists('slug') ? $request->slug : $product->slug,
            'content' => $request->content,
            'warnings' => $request->exists('warnings') ? $request->warnings : $product->warnings,

            'images' => $request->exists('images') ? serialize($request->images) : $product->images,

            'attrs' => $request->exists('attrs') ? serialize($request->attrs) : null,

            'seo_title' => $request->exists('seo_title') ? $request->seo_title : $product->seo_title,
            'seo_description' => $request->exists('seo_description') ? $request->seo_description : $product->seo_description,
            'seo_keywords' => $request->exists('seo_keywords') ? $request->seo_keywords : $product->seo_keywords,
            'seo_canonical' => $request->exists('seo_canonical') ? $request->seo_canonical : $product->seo_canonical,

            'question_status' => $request->question_status,
            'comment_status' => $request->comment_status,

            'status' => 'awaiting',
        ]);

        try {
            /** update specifications */
            if ($request->exists('specifications') && is_array($request->specifications)) {
                foreach ($request->specifications as $specification) {
                    foreach ($specification['keys'] as $key) {
                        $spValueExists = $product->specificationValues()->where('id', $key['value_id'])->first();
                        if ($spValueExists != null) {
                            if ($key['value'] != null) {
                                $spValueExists->update([
                                    'value' => $key['value'],
                                ]);
                            } else {
                                $spValueExists->delete();
                            }
                        } else {
                            if ($key['value'] != null) {
                                $product->specificationValues()->create([
                                    'specification_key_id' => $key['id'],
                                    'value' => $key['value'],
                                ]);
                            }
                        }
                    }
                }
            }
        } catch (Exception $e) {
        }

        /** create and sync tags */
        if ($request->exists('tags') && $request->tags != null && $request->tags != '') {
            $tags = [];
            if (is_array($request->tags)) {
                foreach ($request->tags as $tag) {
                    $findTag = ProductTag::where('title', $tag)->get()->first();
                    if ($findTag != null) {
                        $tags[] = $findTag->id;
                    } else {
                        $createdTag = ProductTag::create([
                            'title' => $tag,
                            'status' => 'publish',
                        ]);
                        $tags[] = $createdTag->id;
                    }
                }
            } else {
                foreach (json_decode($request->tags) as $tag) {
                    $findTag = ProductTag::where('title', $tag->value)->get()->first();
                    if ($findTag != null) {
                        $tags[] = $findTag->id;
                    } else {
                        $createdTag = ProductTag::create([
                            'title' => $tag->value,
                            'status' => 'publish',
                        ]);
                        $tags[] = $createdTag->id;
                    }
                }
            }


            $product->productTags()->sync($tags);
        }

        /** Report Action */
        $reportMessage = __("messages.report_product_updated");
        $this->report($reportMessage, 'product', $product);
        /******************/

        return redirect()->back()->with('message', [__('messages.product_updated')]);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Product $product)
    {
        /** @var App\Models\User $user */
        $user = auth()->user();
        $store = $user->store;

        if ($product->status == 'publish' || $product->status == 'draft') {
            return redirect()->route('seller.products.index')->withErrors([__('messages.unauthorized')]);
        }

        if ($product->user_id != $user->id) {
            return redirect()->route('seller.products.index')->withErrors([__('messages.unauthorized')]);
        }

        if ($product->status == 'rejected' || ($product->status == 'awaiting' && $product->reason == 'create')) {
            // delete product from users cart
            $product->consignmentItems()->whereHas('consignment', function ($query) {
                $query->whereHas('order', function ($query) {
                    $query->whereDoesntHave('orderGroup');
                });
            })->delete();

            if ($product->consignmentItems()->count() > 0) {
                $product->update([
                    'status' => 'deep_deleted'
                ]);
            } else {
                $product->delete();
            }

            return redirect()->route('seller.products.index')->with('message', [__('messages.product_deleted')]);
        }

        return redirect()->route('seller.products.index')->withErrors([__('messages.unauthorized')]);
    }

    /**
     * add product to seller list
     */
    public function add(Product $product)
    {
        /** @var App\Models\User $user */
        $user = auth()->user();
        $store = $user->store;

        $store->products()->attach($product->id);

        return redirect()->route('seller.products.index')->with('message', [__('messages.product_added')]);
    }

    /**
     * add product to seller list
     */
    public function delete(Product $product)
    {
        /** @var App\Models\User $user */
        $user = auth()->user();
        $store = $user->store;

        // prevent remove if has price
        if ($product->inventories()->where('store_id', $store->id)->count() > 0) {
            return redirect()->route('seller.products.index')->withErrors([__('messages.unauthorized')]);
        }

        // prevent remove if seller is creator
        if ($product->user_id == $user->id) {
            return redirect()->route('seller.products.index')->withErrors([__('messages.unauthorized')]);
        }

        $store->products()->detach($product->id);

        return redirect()->route('seller.products.index')->with('message', [__('messages.product_deleted')]);
    }

    /** search raw brand */
    public function searchBrand(Request $request)
    {
        $brands = Brand::query()->where('status', 'publish')->when($request->input('search'), function ($query, $search) {
            $query->where('title', 'like', "%{$search}%");
        })->orderBy('created_at', 'desc')->pluck('title')->take(50)->toArray();

        return response($brands);
    }
}
