<?php

namespace App\Http\Controllers\Store\Products;

use App\Http\Controllers\Controller;
use App\Models\Question;
use App\Models\QuestionAnswer;
use Illuminate\Http\Request;
use Inertia\Inertia;

class QuestionsController extends Controller
{
    /******************** Questions *********************/
    /**
     * questions list
     */
    public function index(Request $request)
    {
        /** @var App\Models\User $user */
        $user = auth()->user();
        $store = $user->store;

        if ($store->direct_post && !$store->transportation_status) {
            return redirect()->route('seller.transportation.index')->withErrors([__('messages.post_ways_required')]);
        }

        /** @var LengthAwarePaginator */
        $questions = Question::query()->whereHas('product', function ($query) use ($store) {
            $query->whereHas('stores', function ($query) use ($store) {
                $query->where('id', $store->id);
            });
        })->when($request->input('publish'), function ($query, $publish) {
            if ($publish) {
                $query->where('status', 'publish')->orderBy('created_at', 'desc');
            }
        })->when($request->input('awaiting_questions'), function ($query, $awaiting_questions) {
            if ($awaiting_questions) {
                $query->where('status', 'awaiting')->orderBy('created_at', 'asc');
            }
        })->when($request->input('awaiting_answers'), function ($query, $awaiting_answers) {
            if ($awaiting_answers) {
                $query->whereHas('questionAnswers', function ($query) {
                    $query->where('status', 'awaiting');
                });
            }
        })->orderByRaw("FIELD(status , 'awaiting', 'publish') ASC")
            ->orderBy('questions.created_at', 'desc')->paginate(8)->through(fn($question) => [
                'id' => $question->id,
                'question' => $question->question,
                'jalali_created_at' => $question->jalali_created_at,
                'status' => $question->status,
                'answers' => $question->questionAnswers->map(fn($answer) => [
                    'id' => $answer->id,
                    'answer' => $answer->answer,
                    'vote_up' => $answer->vote_up,
                    'vote_down' => $answer->vote_down,
                    'is_buyer' => $answer->is_buyer,
                    'status' => $answer->status,
                    'user' => [
                        'id' => $answer->user->id,
                        'get_name' => $answer->user->get_name,
                        'type' => $answer->user->type,
                        'seller' => $answer->user->seller,
                    ],
                ]),
                'product' => [
                    'id' => $question->product->id,
                    'title' => $question->product->title,
                    'slug' => $question->product->slug,
                ],
                'user' => [
                    'id' => $question->user->id,
                    'get_name' => $question->user->get_name,
                ],
            ]);

        $questions->withQueryString();

        return Inertia::render('Store/Products/Questions/QuestionList', [
            'questions' => $questions
        ]);
    }

    /**
     * accept question
     */
    public function question_accept(Question $question)
    {
        if ($question->status == 'awaiting') {
            $question->update([
                'status' => 'publish',
            ]);

            /** Report Action */
            $reportMessage = __("messages.report_question_accepted");
            $this->report($reportMessage, 'product', $question->product);
            /******************/

            return redirect()->back()->with('message', [__('messages.question_accepted')]);
        }

        return redirect()->back();
    }

    /**
     * delete question
     */
    public function question_delete(Question $question)
    {
        if ($question->status == 'awaiting') {
            $question->delete();

            /** Report Action */
            $reportMessage = __("messages.report_question_deleted");
            $this->report($reportMessage, 'product', $question->product);
            /******************/

            return redirect()->back()->with('message', [__('messages.question_deleted')]);
        } else {
            return redirect()->back();
        }
    }

    /******************** Answers *********************/
    /**
     * accept question
     */
    public function answer_accept(QuestionAnswer $answer)
    {
        if ($answer->status == 'awaiting') {
            $answer->update([
                'status' => 'publish',
            ]);

            /** Report Action */
            $reportMessage = __("messages.report_question_answer_accepted");
            $this->report($reportMessage, 'product', $answer->question->product);
            /******************/

            return redirect()->back()->with('message', [__('messages.question_answer_accepted')]);
        }

        return redirect()->back();
    }

    /**
     * send question answer
     */
    public function answer_send(Request $request, Question $question)
    {
        $request->validate([
            'answer' => 'required'
        ]);

        /** @var App\Models\User $user */
        $user = auth()->user();

        $question->questionAnswers()->create([
            'user_id' => $user->id,
            'answer' => $request->answer,
            'status' => 'publish'
        ]);

        /** Report Action */
        $reportMessage = __("messages.report_question_answer_send");
        $this->report($reportMessage, 'product', $question->product);
        /******************/

        return redirect()->back()->with('message', [__('messages.question_answer_send')]);
    }

    /**
     * delete question answer
     */
    public function answer_delete(QuestionAnswer $answer)
    {
        /** @var App\Models\User $user */
        $user = auth()->user();
        $store = $user->store;

        if ($answer->status == 'awaiting' || $answer->user->id == $user->id) {
            $answer->delete();

            /** Report Action */
            $reportMessage = __("messages.report_question_answer_deleted");
            $this->report($reportMessage, 'product', $answer->question->product);
            /******************/

            return redirect()->back()->with('message', [__('messages.question_answer_deleted')]);
        } else {
            return redirect()->back();
        }
    }
}
