<?php

namespace App\Http\Controllers\User\Account;

use App\Helpers\DropdownListHelper;
use App\Http\Controllers\Controller;
use App\Http\Requests\User\ProfileUpdateRequest;
use App\Models\Token;
use App\Services\User\Account\ProfileService;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Session;
use Inertia\Inertia;
use Inertia\Response;

class ProfileController extends Controller
{
    use DropdownListHelper;

    /**
     * edit profile
     */
    public function edit(): Response
    {
        return Inertia::render('User/Account/Edit', [
            'user' => auth()->user(),
            'genderList' => $this->genderList()
        ]);
    }

    /**
     * update profile
     */
    public function update(ProfileUpdateRequest $request, ProfileService $profileService): RedirectResponse
    {
        /** @var \App\Models\User $user */
        $user = auth()->user();

        $user->update([
            'name' => $request->name,
            'username' => $request->exists('username') && $request->username != null ? $request->username : $user->username,
            'email' => $user->email_verified_at == null ? $request->email : $user->email,
            'avatar' => $profileService->uploadAvatar($request, $user),
            'national_number' => $request->exists('national_number') ? $request->national_number : null,
            'birthday' => $profileService->getBirthday($request),
            'gender' => $profileService->getGender($request),
            'bio' => $request->exists('bio') ? $request->bio : null,
            'social' => $profileService->getSocial($request),
        ]);

        return redirect()->back()->with('message', [__('messages.profile_updated')]);
    }

    /**
     * active sessions list
     */
    public function sessions(ProfileService $profileService): Response
    {
        /** @var \App\Models\User $user */
        $user = auth()->user();

        // user sessions
        $sessions = $profileService->getSessions($user);

        // current session
        $currentSesison = $user->tokens()->where('session', session()->getId())->first();

        return Inertia::render('User/Account/Sessions', [
            'sessions' => $sessions,
            'current_session' =>  $currentSesison != null ?  $currentSesison->id : null
        ]);
    }

    /**
     * terminate user session
     * @method delete
     */
    public function terminate(Request $request): RedirectResponse
    {
        $request->validate([
            'session_id' => ['required']
        ]);

        /** @var App\Models\User $user */
        $user = auth()->user();

        $session = Token::where('id', $request->session_id)->where('user_id', $user->id)->first();

        if ($session != null) {

            if ($session->session == session()->getId()) {
                Auth::logout();

                $request->session()->invalidate();

                $request->session()->regenerateToken();

                $session->delete();

                return redirect()->route('index')->with('message', [__('messages.session_terminated')]);
            }

            $result = Session::getHandler()->destroy($session->session);

            if ($result) {
                $session->delete();
                return redirect()->back()->with('message', [__('messages.session_terminated')]);
            }
            return redirect()->back()->withErrors(['error' => __('messages.unknown_error')]);
        }

        return redirect()->back()->withErrors([__('messages.invalid_session')]);
    }
}
