<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Support\Facades\Cache;
use Morilog\Jalali\Jalalian;

class Consignment extends Model
{
    use HasFactory;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'id',
        'order_id',
        'user_id',
        'store_id',
        'uuid',
        'send_method',
        'send_type',
        'address',
        'tracking_code',
        'status',
    ];

    protected $casts = [
        'order_id' => 'integer',
        'user_id' => 'integer',
        'store_id' => 'integer',
    ];

    /**
     * additional data
     */
    protected $appends = [
        'is_all_product_sent',
        'has_awaiting_request_from_seller',
        'is_products_awaiting_receipt_from_seller',

        'get_send_method',
        'get_address',
        'get_shares',

        'get_total_bill',
        'get_total_cancel_bill',
        'get_total_final_bill',

        'jalali_created_at',
        'jalali_updated_at',
    ];

    public function getIsAllProductSentAttribute()
    {
        return $this->consignmentItems()->where('status', 'awaiting_seller')->count() == 0;
    }

    public function getHasAwaitingRequestFromSellerAttribute()
    {
        return $this->consignmentItems()->where('status', 'awaiting_request_from_seller')->count() > 0;
    }

    public function getIsProductsAwaitingReceiptFromSellerAttribute()
    {
        return $this->consignmentItems()->where('status', 'awaiting_receipt_from_seller')->count() > 0;
    }

    public function getGetSendMethodAttribute()
    {
        return $this->send_method != null ? unserialize($this->send_method) : null;
    }

    public function getGetAddressAttribute()
    {
        return $this->address != null ? unserialize($this->address) : null;
    }

    public function getGetSharesAttribute()
    {
        $consignmentItems = $this->consignmentItems()->where('status', '!=', 'cancelled')->where('status', '!=', 'returned')->get();

        return [
            'shop' => $consignmentItems->sum('get_shares.shop'),
            'seller' => $consignmentItems->sum('get_shares.seller'),
            'affiliate' => $consignmentItems->sum('get_shares.affiliate'),
        ];
    }

    public function getGetTotalBillAttribute()
    {
        $consignmentItems = $this->consignmentItems()->get();

        return [
            'price' => $consignmentItems->sum('get_full_count_price'),
            'discount' => $consignmentItems->sum('get_full_count_discount'),
            'count' => $consignmentItems->sum('count'),
        ];
    }

    public function getGetTotalCancelBillAttribute()
    {
        $consignmentItems = $this->consignmentItems()->where(function ($query) {
            $query->where('status', 'cancelled')->orWhere('status', 'returned');
        })->get();

        return [
            'price' => $consignmentItems->sum('get_full_count_price'),
        ];
    }

    public function getGetTotalFinalBillAttribute()
    {
        $consignmentItems = $this->consignmentItems()->where('status', '!=', 'cancelled')->where('status', '!=', 'returned')->get();

        return [
            'price' => $consignmentItems->sum('get_full_count_price'),
            'discount' => $consignmentItems->sum('get_full_count_discount'),
            'count' => $consignmentItems->sum('count'),
        ];
    }

    public function getJalaliCreatedAtAttribute()
    {
        $date = [
            Jalalian::forge($this->created_at)->format('%d %B %Y'),
            Jalalian::forge($this->created_at)->format('Y/m/d'),
            Jalalian::forge($this->created_at)->format('H:i - Y/m/d'),
            Jalalian::forge($this->created_at)->ago(),
            Jalalian::forge($this->created_at)->getTimestamp(),
        ];
        return $date;
    }

    public function getJalaliUpdatedAtAttribute()
    {
        $date = [
            Jalalian::forge($this->updated_at)->format('%d %B %Y'),
            Jalalian::forge($this->updated_at)->format('Y/m/d'),
            Jalalian::forge($this->updated_at)->format('H:i - Y/m/d'),
            Jalalian::forge($this->updated_at)->ago(),
            Jalalian::forge($this->updated_at)->getTimestamp(),
        ];
        return $date;
    }

    public function getSafeAttribute()
    {
        return [
            'id' => $this->id,
            'uuid' => $this->uuid,
            'status' => $this->status,
            'jalali_created_at' => $this->jalali_created_at,
            'jalali_updated_at' => $this->jalali_updated_at,
        ];
    }
    /** end append */

    /**
     * remove cache data in update
     */
    protected static function boot()
    {
        parent::boot();

        static::saved(function ($consignment) {
            if ($consignment->store_id != null)
                Cache::forget('store_consignments_count_' . $consignment->store_id);
        });

        static::deleted(function ($consignment) {
            if ($consignment->store_id != null)
                Cache::forget('store_consignments_count_' . $consignment->store_id);
        });
    }

    /* relationships **************/
    public function order(): BelongsTo
    {
        return $this->belongsTo(Order::class);
    }

    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    public function store(): BelongsTo
    {
        return $this->belongsTo(Store::class);
    }

    public function consignmentItems(): HasMany
    {
        return $this->hasMany(ConsignmentItem::class);
    }
}
