<?php

namespace App\Models;

use Carbon\Carbon;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Morilog\Jalali\Jalalian;

class Discount extends Model
{
    use HasFactory;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'user_id',

        'for',
        'title',
        'code',
        'expire_at',
        'percent',
        'max_amount',
        'max_use',
        'status',
    ];

    protected $casts = [
        'percent' => 'integer',
        'max_amount' => 'integer',
        'max_use' => 'integer',
    ];

    /**
     * additional data
     */
    protected $appends = [
        'use_count',
        'get_for',
        'get_formatted_max_amount',
        'is_active',
        'jalali_expire_at',
        'jalali_created_at',
        'jalali_updated_at',
    ];

    public function getUseCountAttribute()
    {
        return number_format($this->users()->get()->count());
    }

    public function getGetForAttribute()
    {
        $for = null;
        if ($this->for == 'all') {
            $for = 'all';
        } else {
            $user = User::where('id', $this->for)->first();
            if ($user != null) {
                $for = [
                    'id' => $user->id,
                    'name' => $user->name,
                    'username' => $user->username,
                    'phone' => $user->phone,
                ];
            } else {
                $for = 'nobody';
            }
        }
        return $for;
    }

    public function getGetFormattedMaxAmountAttribute()
    {
        return number_format($this->max_amount);
    }

    public function getIsActiveAttribute()
    {
        return $this->expire_at > Carbon::now() ? true : false;
    }

    public function getJalaliExpireAtAttribute()
    {
        $date = [
            Jalalian::forge($this->expire_at)->format('%d %B %Y'),
            Jalalian::forge($this->expire_at)->format('Y/m/d'),
            Jalalian::forge($this->expire_at)->format('H:i - Y/m/d'),
            Jalalian::forge($this->expire_at)->ago(),
            Jalalian::forge($this->expire_at)->getTimestamp(),
        ];
        return $date;
    }

    public function getJalaliCreatedAtAttribute()
    {
        $date = [
            Jalalian::forge($this->created_at)->format('%d %B %Y'),
            Jalalian::forge($this->created_at)->format('Y/m/d'),
            Jalalian::forge($this->created_at)->format('H:i - Y/m/d'),
            Jalalian::forge($this->created_at)->ago(),
            Jalalian::forge($this->created_at)->getTimestamp(),
        ];
        return $date;
    }

    public function getJalaliUpdatedAtAttribute()
    {
        $date = [
            Jalalian::forge($this->updated_at)->format('%d %B %Y'),
            Jalalian::forge($this->updated_at)->format('Y/m/d'),
            Jalalian::forge($this->updated_at)->format('H:i - Y/m/d'),
            Jalalian::forge($this->updated_at)->ago(),
            Jalalian::forge($this->updated_at)->getTimestamp(),
        ];
        return $date;
    }
    /** end append */

    /* relationships **************/
    public function users(): BelongsToMany
    {
        return $this->belongsToMany(User::class);
    }
}
