<?php

namespace App\Models;

use Cviebrock\EloquentSluggable\Sluggable;
use DateTimeZone;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\HasOne;
use Illuminate\Support\Facades\Cache;
use Morilog\Jalali\Jalalian;

class Product extends Model
{
    use HasFactory, Sluggable;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'user_id',
        'edited_by',
        'brand_id',
        'category_id',

        'uuid',
        'title',
        'title_en',
        'slug',
        'content',
        'warnings',
        'guide',
        'tabs',

        'seo_title',
        'seo_description',
        'seo_keywords',
        'seo_canonical',

        'images',
        'videos',
        'model_3d',

        'attrs',

        'view_count',

        'in_stock_status',
        'inquiry_options',

        'commission',

        'is_vip',
        'comment_status',
        'question_status',

        'reason',
        'status',
    ];

    protected $casts = [
        'user_id' => 'integer',
        'edited_by' => 'integer',
        'brand_id' => 'integer',
        'category_id' => 'integer',

        'view_count' => 'integer',

        'in_stock_status' => 'boolean',

        'commission' => 'double',

        'is_vip' => 'boolean',
        'comment_status' => 'boolean',
        'question_status' => 'boolean',
    ];

    /**
     * additional data
     */
    protected $appends = [
        'category',
        'favorites_count',
        'cart_count',
        'get_simple_brand',
        'get_images_link',
        'get_images_id',
        'get_all_images',
        'get_buyer_images',
        'get_images',
        'get_videos',
        'get_attrs',
        'get_tabs',
        'get_specifications',
        'get_tags',
        'get_tags_string',
        'get_formatted_view_count',
        'get_inquiry_options',
        'price_model',
        'raw_price_model',
        'inventory_status',
        'inventory',
        'has_discount',
        'best_price',
        'best_normal_price',
        'unique_color',
        'best_inventory',
        'comments_info',
        'questions_info',
        'answers_info',
        'get_seo_keywords',
        'get_affiliate',
        'jalali_created_at',
        'jalali_updated_at',

        // base on  auth user
        'is_favorite',
        'is_buyer',
        'has_inventory_notification',
        'has_discount_notification',

        'safe',
    ];

    public function getCategoryAttribute()
    {
        return $this->productCategory != null ? $this->productCategory->simple_safe : null;
    }

    public function getFavoritesCountAttribute()
    {
        return $this->productFavorites()->count();
    }

    public function getCartCountAttribute()
    {
        $ConsignmentItemInCart = Cache::remember('product_cart_count_' . $this->id, now()->addMinutes(5), function () {
            return ConsignmentItem::where('product_id', $this->id)
                ->whereHas('consignment.order', function ($query) {
                    $query->whereDoesntHave('orderGroup');
                })
                ->sum('count');
        });

        return $ConsignmentItemInCart;
    }

    public function getGetSimpleBrandAttribute()
    {
        return Cache::remember('product_simple_brand_' . $this->brand_id, now()->addMonth(), function () {
            return $this->brand != null ? [
                'id' => $this->brand->id,
                'uuid' => $this->brand->uuid,
                'title' => $this->brand->title,
                'title_en' => $this->brand->title_en,
                'slug' => $this->brand->slug,
            ] : null;
        });
    }

    public function getGetImagesLinkAttribute()
    {
        $images = [];
        if ($this->images != null && is_array(unserialize($this->images))) {
            foreach (unserialize($this->images) as $image) {
                $file = Cache::remember('files_' . $image, now()->addMonth(), function () use ($image) {
                    return File::find($image);
                });
                if ($file != null && $file->type == "image") {
                    $images[] = asset($file->url);
                }
            }
        }
        return $images;
    }

    public function getGetImagesIdAttribute()
    {
        $imagesId = [];
        if ($this->images != null && is_array(unserialize($this->images))) {
            $imagesId = unserialize($this->images);
        }
        return $imagesId;
    }

    public function getGetAllImagesAttribute()
    {
        return Cache::remember('product_all_images_' . $this->id, now()->addHour(), function () {
            $images = [];
            if ($this->images != null && is_array(unserialize($this->images))) {
                foreach (unserialize($this->images) as $image) {
                    $file = Cache::remember('files_' . $image, now()->addMonth(), function () use ($image) {
                        return File::find($image);
                    });
                    if ($file != null && $file->type == "image") {
                        $images[] = [
                            'link' => asset($file->url),
                            'alt' => $file->description,
                        ];
                    }
                }
            }
            foreach ($this->inventories as $inventory) {
                if ($inventory->get_image['id'] != null) {
                    $images[] = [
                        'link' => $inventory->get_image['url'],
                        'alt' => $inventory->get_image['alt'],
                    ];
                }
            }

            if (count($images) == 0) {
                $images[] = [
                    'link' => asset('img/temp/pre-load-product.png'),
                    'alt' => __('messages.word.no_image'),
                ];
            }

            return $images;
        });
    }

    public function getGetBuyerImagesAttribute()
    {
        $images = [];
        $comments = $this->productComments()->where('images', '!=', null)->where('status', 'publish')->get();
        foreach ($comments as $comment) {
            foreach ($comment->get_images as $cimage) {
                $images[] = [
                    'link' => $cimage['url'],
                    'alt' => __('messages.sentence.image_sent_by_user_', ['name' => $cimage['alt']]),
                ];
            }
        }

        return $images;
    }

    public function getGetImagesAttribute()
    {
        $images = [];
        if ($this->images != null && is_array(unserialize($this->images))) {
            foreach (unserialize($this->images) as $image) {
                $file = Cache::remember('files_' . $image, now()->addMonth(), function () use ($image) {
                    return File::find($image);
                });
                if ($file != null && $file->type == "image") {
                    $images[] = [
                        'link' => asset($file->url),
                        'alt' => $file->description,
                    ];
                    return $images;
                }
            }
        }
        if (count($images) == 0) {
            $images[] = [
                'link' => asset('img/temp/pre-load-product.png'),
                'alt' => __('messages.word.no_image'),
            ];
        }
        return $images;
    }

    public function getGetVideosAttribute()
    {
        return $this->videos != null && $this->videos != "a:0:{}" ? unserialize($this->videos) : [
            "type" => null,
            "link" => null,
            "embed" => null
        ];
    }

    public function getGetAttrsAttribute()
    {
        return $this->attrs != null ? unserialize($this->attrs) : [];
    }

    public function getGetTabsAttribute()
    {
        return $this->tabs != null ? unserialize($this->tabs) : [];
    }

    public function getGetSpecificationsAttribute()
    {
        return Cache::remember('specifications_' . $this->category_id, now()->addMonth(), function () {
            return $this->productCategory->specifications()->get();
        })->map(fn($specification) => [
            'id' => $specification->id,
            'title' => $specification->title,
            'keys' => Cache::remember('specification_keys_' . $specification->id, now()->addMonth(), function () use ($specification) {
                return $specification->specificationKeys()->get();
            })->map(fn($specificationKey) => [
                'id' => $specificationKey->id,
                'key' => $specificationKey->key,
                'value' => Cache::remember('product_' . $this->id . '_specification_key_values_' . $specificationKey->id, now()->addMonth(), function () use ($specificationKey) {
                    return $specificationKey->specificationValues()->where('product_id', $this->id)->first() != null ? $specificationKey->specificationValues()->where('product_id', $this->id)->first()->value : $specificationKey->default_value;
                }),
            ])->toArray(),
        ])->toArray();
    }

    public function getGetInquiryOptionsAttribute()
    {
        if ($this->inquiry_options != null && is_array(unserialize($this->inquiry_options))) {
            return unserialize($this->inquiry_options);
        }
        return null;
    }

    public function getGetTagsAttribute()
    {
        return $this->productTags()->pluck('title')->toArray();
    }

    public function getGetTagsStringAttribute()
    {
        $tags = [];
        foreach ($this->productTags()->pluck('title')->toArray() as $tag) {
            $tags[] = [
                'value' => $tag
            ];
        }
        return json_encode($tags, \JSON_UNESCAPED_UNICODE);
    }

    public function getGetFormattedViewCountAttribute()
    {
        return number_format($this->view_count);
    }

    public function getPriceModelAttribute()
    {
        $priceModel = [];

        $pmArray = Cache::remember('product_props_' . $this->id, now()->addMonth(), function () {
            return $this->productProps()->get()->map(fn($props) => [
                'id' => $props->id,
                'child' => $props->child,
                'name' => $props->name,
                'type' => $props->type,
            ])->toArray();
        });

        foreach ($pmArray as $pm) {
            $priceModel[$pm['id']] = [
                'child' => $pm['child'],
                'name' => $pm['name'],
                'type' => $pm['type'],
            ];
        }

        return  $priceModel;
    }

    public function getRawPriceModelAttribute()
    {
        $priceModel = [];

        $priceModel = Cache::remember('product_raw_props_' . $this->id, now()->addMonth(), function () {
            return $this->productProps()->get()->map(fn($props) => [
                'name' => $props->name,
                'type' => $props->type,
                'active' => true,
            ])->toArray();
        });

        return  $priceModel;
    }

    public function getInventoryStatusAttribute()
    {
        return Cache::remember('product_inventory_status_' . $this->id, now()->addMonth(), function () {
            return [
                'inventory' => $this->inventories()->where('status', 'publish')->where(function ($query) {
                    $query->doesntHave('store')->orWhereHas('store', function ($query) {
                        $query->where('status', 'active');
                    });
                })->exists(),
                'instock' => $this->inventories()->where('status', 'publish')->where('count', '>', 0)->where(function ($query) {
                    $query->doesntHave('store')->orWhereHas('store', function ($query) {
                        $query->where('status', 'active');
                    });
                })->exists(),
                'inquery' => !$this->in_stock_status
            ];
        });
    }

    public function getInventoryAttribute()
    {
        return Cache::remember('product_inventory_' . $this->id, now()->addMonth(), function () {
            return $this->inventories()->where('status', 'publish')->where('count', '>', 0)->where(function ($query) {
                $query->doesntHave('store')->orWhereHas('store', function ($query) {
                    $query->where('status', 'active');
                });
            })->exists();
        });
    }

    public function getHasDiscountAttribute()
    {
        return $this->inventories()->where('status', 'publish')->where('count', '>', 0)->where('discount_price', '!=', null)->where(function ($query) {
            $query->doesntHave('store')->orWhereHas('store', function ($query) {
                $query->where('status', 'active');
            });
        })->exists();
    }

    public function getBestPriceAttribute()
    {
        return Cache::remember('product_best_price_' . $this->id, now()->addMonth(), function () {
            return $this->inventories()->where('status', 'publish')->where('count', '>', 0)->where(function ($query) {
                $query->doesntHave('store')->orWhereHas('store', function ($query) {
                    $query->where('status', 'active');
                });
            })->get()->min('get_final_price');
        });
    }

    public function getBestNormalPriceAttribute()
    {
        $bestInventory = $this->best_inventory;

        $price = null;
        $discount_price = null;
        $discount_expire = null;
        $percent = null;

        if ($this->inventory && $bestInventory != null) {
            $price = number_format($bestInventory['price']);
            if ($bestInventory['is_discount_valid']) {
                $discount_price = number_format($bestInventory['discount_price']);

                if ($bestInventory['discount_expire'] != null)
                    $discount_expire = Jalalian::forge($bestInventory['discount_expire'], new DateTimeZone("Asia/Tehran"))->getTimestamp();

                $percent = $bestInventory['get_discount_percent'];
            }
        }

        return [
            'price' => $price,
            'discount_price' => $discount_price,
            'discount_expire' => $discount_expire,
            'percent' => $percent
        ];
    }

    public function getUniqueColorAttribute()
    {
        return Cache::remember('product_unique_color_' . $this->id, now()->addMonth(), function () {
            $inventories = $this->inventories()->where('status', 'publish')->where(function ($query) {
                $query->doesntHave('store')->orWhereHas('store', function ($query) {
                    $query->where('status', 'active');
                });
            })->get();

            $mainLevels = [];
            $mainLevelIDs = [];
            foreach ($inventories as $inventory) {
                $mainLevelModel = $inventory->inventoryProps->where('child', 0)->first();

                if ($mainLevelModel != null) {

                    if ($mainLevelModel->type != 'color') {
                        return [];
                    }

                    $mainLevelIDs[$mainLevelModel->value][] = $mainLevelModel->id;
                    $mainLevels[$mainLevelModel->type == 'color' ? unserialize($mainLevelModel->value)['color'] : $mainLevelModel->value] = [
                        'id' => $mainLevelIDs[$mainLevelModel->value],
                        'name' => $mainLevelModel->name,
                        'type' => $mainLevelModel->type,
                        'value' => $mainLevelModel->type == 'color' ? unserialize($mainLevelModel->value) : $mainLevelModel->value,
                    ];
                }
            }

            return $mainLevels;
        });
    }

    public function getBestInventoryAttribute()
    {
        $bestInventory = null;

        if ($this->inventory) {
            $bestPriceNumber = $this->best_price;
            $bestInventories = $this->inventories()->where('status', 'publish')->where('count', '>', 0)->where(function ($query) use ($bestPriceNumber) {
                $query->where('price', $bestPriceNumber)->orWhere('discount_price', $bestPriceNumber);
            })->get();

            $inventoryArray = [];
            foreach ($bestInventories as $inventory) {
                $inventoryArray[$inventory->id] = $inventory->is_discount_valid ? $inventory->discount_price : $inventory->price;
            }

            $bestInventoryID = array_search(min(array_unique($inventoryArray)), array_unique($inventoryArray));

            $bestInventory = $this->inventories()->where('id', $bestInventoryID)->where('status', 'publish')->where('count', '>', 0)->first();
            if ($bestInventory != null) {
                $bestInventory = [
                    'id' => $bestInventory->id,
                    'store' => $bestInventory->store_id != null ? $bestInventory->store->safe : null,
                    'price' => $bestInventory->price,
                    'is_discount_valid' => $bestInventory->is_discount_valid,
                    'discount_price' => $bestInventory->discount_price,
                    'discount_expire' => $bestInventory->discount_expire,
                    'jalali_discount_expire' => $bestInventory->jalali_discount_expire,
                    'get_discount_percent' => $bestInventory->get_discount_percent,
                    'min_sale' => $bestInventory->min_sale,
                    'max_sale' => $bestInventory->max_sale,
                    'raw_props' => $bestInventory->raw_props,
                    'props' => $bestInventory->props,
                    'get_image' => $bestInventory->get_image,
                    'original' => $bestInventory['original'],
                    'used' => $bestInventory->used,
                ];
            }

            return $bestInventory;
        }

        return $bestInventory;
    }

    public function getHasInventoryNotificationAttribute()
    {
        /** @var App\Models\User $user */
        $user = auth()->user();

        if ($user != null) {
            return Cache::remember('product_has_inventory_notification_' . $this->id . '_' . $user->id, now()->addMonth(), function () use ($user) {
                return $user->productNotifications()->where('product_id', $this->id)->where('type', 'inventory')->first() != null ? true : false;
            });
        } else {
            return false;
        }
    }

    public function getHasDiscountNotificationAttribute()
    {
        /** @var App\Models\User $user */
        $user = auth()->user();

        if ($user != null) {
            return Cache::remember('product_has_discount_notification_' . $this->id . '_' . $user->id, now()->addMonth(), function () use ($user) {
                return $user->productNotifications()->where('product_id', $this->id)->where('type', 'discount')->first() != null ? true : false;
            });
        } else {
            return false;
        }
    }

    public function getCommentsInfoAttribute()
    {
        return Cache::remember('product_comments_info_' . $this->id, now()->addMonth(), function () {
            $productComments = $this->productComments()->get();

            $result = [];
            if (count($productComments) > 0) {
                $result = [
                    'count' => number_format(count($productComments)),
                    'rating' => floor($productComments->avg('rating') * 10) / 10,
                    'rating_percent' => ((floor($productComments->avg('rating') * 10) / 10) * 100) / 5,
                    'suggation_percent' => (floor((($productComments->avg('buy_suggest') * 100) / 1) * 10)) / 10,
                ];
            } else {
                $result = [
                    'count' => 0,
                    'rating' => 5,
                    'rating_percent' => 100,
                    'suggation_percent' => 100,
                ];
            }

            return $result;
        });
    }

    public function getQuestionsInfoAttribute()
    {
        return Cache::remember('product_questions_info_' . $this->id, now()->addMonth(), function () {
            return [
                'count' => number_format($this->questions()->count()),
            ];
        });
    }

    public function getAnswersInfoAttribute()
    {
        return Cache::remember('product_answers_info_' . $this->id, now()->addMonth(), function () {
            return [
                'count' => number_format($this->questions()->withCount('questionAnswers')->get()->sum('question_answers_count')),
            ];
        });
    }

    public function getGetSeoKeywordsAttribute()
    {
        $seo_keywords = [];
        foreach (($this->seo_keywords != null ? json_decode($this->seo_keywords) : []) as $keywords) {
            $seo_keywords[] = $keywords->value;
        }

        return implode(',', $seo_keywords);
    }

    public function getGetAffiliateAttribute()
    {
        return Cache::remember('product_affiliate_' . $this->id, now()->addMonth(), function () {
            return $this->affiliate;
        });
    }


    public function getIsFavoriteAttribute()
    {
        /** @var App\Models\User $user */
        $user = auth()->user();

        if ($user != null) {
            $favorite = Cache::remember('product_is_favorite_' . $this->id . '_' . $user->id, now()->addMonth(), function () use ($user) {
                return $user->productFavorites()->where('product_id', $this->id)->first();
            });

            return $favorite != null ? true : false;
        } else {
            return false;
        }
    }

    public function getIsBuyerAttribute()
    {
        /** @var \App\Models\User $user */
        $user = auth()->user();

        if ($user == null) return null;

        return $user->consignments()->whereHas('consignmentItems', function ($query) {
            $query->where('product_id', $this->id)->where(function ($query) {
                $query->where('status', 'sent')->orWhere('status', 'delivered');
            });
        })->count() > 0 ? true : false;
    }

    public function getJalaliCreatedAtAttribute()
    {
        $date = [
            Jalalian::forge($this->created_at)->format('%d %B %Y'),
            Jalalian::forge($this->created_at)->format('Y/m/d'),
            Jalalian::forge($this->created_at)->format('H:i - Y/m/d'),
            Jalalian::forge($this->created_at)->ago(),
            Jalalian::forge($this->created_at)->getTimestamp(),
        ];
        return $date;
    }

    public function getJalaliUpdatedAtAttribute()
    {
        $date = [
            Jalalian::forge($this->updated_at)->format('%d %B %Y'),
            Jalalian::forge($this->updated_at)->format('Y/m/d'),
            Jalalian::forge($this->updated_at)->format('H:i - Y/m/d'),
            Jalalian::forge($this->updated_at)->ago(),
            Jalalian::forge($this->updated_at)->getTimestamp(),
        ];
        return $date;
    }

    public function getSafeAttribute()
    {
        return [
            'id' => $this->id,
            'uuid' => $this->uuid,
            'title' => $this->title,
            'title_en' => $this->title_en,
            'slug' => $this->slug,
            'content' => $this->content,
            'warnings' => $this->warnings,

            'get_images' => $this->get_images,
            'get_videos' => $this->get_videos,
            'get_attrs' => $this->get_attrs,
            'get_specifications' => $this->get_specifications,

            'seo_title' => $this->seo_title,
            'seo_description' => $this->seo_description,
            'seo_keywords' => $this->seo_keywords,
            'get_seo_keywords' => $this->get_seo_keywords,
            'seo_canonical' => $this->seo_canonical,

            'price_model' => $this->price_model,
            'raw_price_model' => $this->raw_price_model,
            'inventory' => $this->inventory,
            'has_discount' => $this->has_discount,
            'best_price' => $this->best_price,
            'best_normal_price' => $this->best_normal_price,
            'unique_color' => $this->unique_color,
            'best_inventory' => $this->best_inventory,

            'is_favorite' => $this->is_favorite,
            'has_inventory_notification' => $this->has_inventory_notification,
            'has_discount_notification' => $this->has_discount_notification,

            'brand' => $this->brand != null ? $this->brand->simple_safe : null,
            'category' => $this->category,
            'get_tags' => $this->get_tags,

            'get_formatted_view_count' => $this->get_formatted_view_count,

            'in_stock_status' => $this->in_stock_status,
            'get_inquiry_options' => $this->get_inquiry_options,

            'is_vip' => $this->is_vip,
            'comment_status' => $this->comment_status,
            'question_status' => $this->question_status,

            'get_affiliate' => $this->get_affiliate,

            'comments_info' => $this->comments_info,
            'questions_info' => $this->questions_info,
            'answers_info' => $this->answers_info,

            'status' => $this->status,

            'jalali_created_at' => $this->jalali_created_at,
            'jalali_updated_at' => $this->jalali_updated_at,

            'tags' => $this->productTags()->where('status', 'publish')->get()->map(fn($tag) => [
                'safe' => $tag->safe
            ])
        ];
    }
    /** end append */

    /**
     * Return the sluggable configuration array for this model.
     *
     * @return array
     */
    public function sluggable(): array
    {
        return [
            'slug' => [
                'source' => 'title'
            ]
        ];
    }

    /**
     * Get the route key for the model.
     *
     * @return string
     */
    public function getRouteKeyName()
    {
        return 'slug';
    }

    /**
     * remove cache data in update
     */
    protected static function boot()
    {
        parent::boot();

        static::saved(function ($product) {
            if (array_key_exists('view_count', $product->getDirty())) return;

            // product all data
            Cache::forget('product_' . $product->slug);

            // user products count
            Cache::forget('products_count_user_' . $product->user_id);

            // article product data
            Cache::forget('article_product_' . $product->uuid);

            // story product data
            Cache::forget('story_product_' . $product->uuid);

            // main custom product lists
            Cache::forget('main_categories_with_product');
            Cache::forget('main_categories_with_product_' . $product->category_id);
            Cache::forget('category_products_list_' . $product->category_id);

            // product brand
            Cache::forget('product_simple_brand_' . $product->brand_id);

            // product props
            Cache::forget('product_props_' . $product->id);
            Cache::forget('product_raw_props_' . $product->id);

            // product all images
            Cache::forget('product_all_images_' . $product->id);
            Cache::forget('product_buyer_images_' . $product->id);

            // product inventory status
            Cache::forget('product_best_inventory_' . $product->id);
            Cache::forget('product_inventory_status_' . $product->id);
            Cache::forget('get_product_inventories_' . $product->id);
        });

        static::deleted(function ($product) {
            // product all data
            Cache::forget('product_' . $product->slug);

            // user products count
            Cache::forget('products_count_user_' . $product->user_id);

            // article product data
            Cache::forget('article_product_' . $product->uuid);

            // story product data
            Cache::forget('story_product_' . $product->uuid);

            // main custom product lists
            Cache::forget('main_categories_with_product');
            Cache::forget('main_categories_with_product_' . $product->category_id);
            Cache::forget('category_products_list_' . $product->category_id);

            // product brand
            Cache::forget('product_simple_brand_' . $product->brand_id);

            // product props
            Cache::forget('product_props_' . $product->id);
            Cache::forget('product_raw_props_' . $product->id);

            // product all images
            Cache::forget('product_all_images_' . $product->id);
            Cache::forget('product_buyer_images_' . $product->id);

            // product inventory status
            Cache::forget('product_best_inventory_' . $product->id);
            Cache::forget('product_inventory_status_' . $product->id);
            Cache::forget('get_product_inventories_' . $product->id);
        });
    }

    /* relationships **************/
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    public function editedBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'edited_by');
    }

    public function inventories(): HasMany
    {
        return $this->hasMany(Inventory::class);
    }

    public function brand(): BelongsTo
    {
        return $this->belongsTo(Brand::class);
    }

    public function productCategory(): BelongsTo
    {
        return $this->belongsTo(ProductCategory::class, 'category_id');
    }

    public function productTags(): BelongsToMany
    {
        return $this->belongsToMany(ProductTag::class);
    }

    public function specificationValues(): HasMany
    {
        return $this->hasMany(SpecificationValue::class);
    }

    public function productFavorites(): HasMany
    {
        return $this->hasMany(ProductFavorite::class);
    }

    public function stores(): BelongsToMany
    {
        return $this->belongsToMany(Store::class);
    }

    public function productProps(): HasMany
    {
        return $this->hasMany(ProductProp::class);
    }

    public function productNotifications(): HasMany
    {
        return $this->hasMany(ProductNotification::class);
    }

    public function productComments(): HasMany
    {
        return $this->hasMany(ProductComment::class);
    }

    public function questions(): HasMany
    {
        return $this->hasMany(Question::class);
    }

    public function affiliate(): HasOne
    {
        return $this->hasOne(Affiliate::class);
    }

    public function consignmentItems(): HasMany
    {
        return $this->hasMany(ConsignmentItem::class);
    }
}
