<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Support\Facades\Cache;
use Morilog\Jalali\Jalalian;

class Question extends Model
{
    use HasFactory;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'user_id',
        'product_id',

        'question',

        'status',
    ];

    protected $casts = [
        'user_id' => 'integer',
        'product_id' => 'integer',
    ];

    /**
     * additional data
     */
    protected $appends = [
        'answers',
        'full_answers',
        'jalali_created_at',
        'jalali_updated_at',

        'safe',
    ];

    public function getAnswersAttribute()
    {
        return Cache::remember('question_answers_' . $this->id, now()->addMonth(), function () {
            return $this->questionAnswers()->orderByRaw("FIELD(status , 'awaiting', 'publish') ASC")->orderBy('created_at', 'desc')->get()->map(fn($answer) => [
                'safe' => $answer->safe
            ]);
        });
    }

    public function getFullAnswersAttribute()
    {
        return $this->questionAnswers()->orderByRaw("FIELD(status , 'awaiting', 'publish') ASC")->orderBy('created_at', 'desc')->get();
    }

    public function getJalaliCreatedAtAttribute()
    {
        $date = [
            Jalalian::forge($this->created_at)->format('%d %B %Y'),
            Jalalian::forge($this->created_at)->format('Y/m/d'),
            Jalalian::forge($this->created_at)->format('H:i - Y/m/d'),
            Jalalian::forge($this->created_at)->ago(),
            Jalalian::forge($this->created_at)->getTimestamp(),
        ];
        return $date;
    }

    public function getJalaliUpdatedAtAttribute()
    {
        $date = [
            Jalalian::forge($this->updated_at)->format('%d %B %Y'),
            Jalalian::forge($this->updated_at)->format('Y/m/d'),
            Jalalian::forge($this->updated_at)->format('H:i - Y/m/d'),
            Jalalian::forge($this->updated_at)->ago(),
            Jalalian::forge($this->updated_at)->getTimestamp(),
        ];
        return $date;
    }

    public function getSafeAttribute()
    {
        return [
            'id' => $this->id,
            'question' => $this->question,
            'status' => $this->status,
            'jalali_created_at' => $this->jalali_created_at,
            'jalali_updated_at' => $this->jalali_updated_at,

            'user' => $this->user->safe,
            'product' => [
                'id' => $this->product->id,
                'title' => $this->product->title,
                'slug' => $this->product->slug,
            ],
            'answers' => $this->answers
        ];
    }
    /** end append */

    /**
     * remove cache data in update
     */
    protected static function boot()
    {
        parent::boot();

        static::saved(function ($question) {
            Cache::forget('product_questions_info_' . $question->product_id);
        });

        static::deleted(function ($question) {
            Cache::forget('product_questions_info_' . $question->product_id);
        });
    }

    /* relationships **************/
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    public function product(): BelongsTo
    {
        return $this->belongsTo(Product::class);
    }

    public function questionAnswers(): HasMany
    {
        return $this->hasMany(QuestionAnswer::class);
    }
}
