<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Support\Facades\Cache;
use Morilog\Jalali\Jalalian;

class QuestionAnswer extends Model
{
    use HasFactory;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'user_id',
        'question_id',

        'answer',
        'vote_up',
        'vote_down',

        'status',
    ];

    protected $casts = [
        'user_id' => 'integer',
        'question_id' => 'integer',
    ];

    /**
     * additional data
     */
    protected $appends = [
        'is_buyer',
        'jalali_created_at',
        'jalali_updated_at',

        'safe',
    ];

    public function getIsBuyerAttribute()
    {
        $user = $this->user;
        $isBuyer = Cache::remember('product_question_answer_is_buyer_' . $this->id . '_' . $user->id, now()->addWeek(), function () use ($user) {
            return $user->consignments()->whereHas('consignmentItems', function ($query) {
                $query->where('product_id', $this->question->product_id)->where(function ($query) {
                    $query->where('status', 'sent')->orWhere('status', 'delivered');
                });
            })->count() > 0 ? true : false;
        });

        return $isBuyer;
    }

    public function getJalaliCreatedAtAttribute()
    {
        $date = [
            Jalalian::forge($this->created_at)->format('%d %B %Y'),
            Jalalian::forge($this->created_at)->format('Y/m/d'),
            Jalalian::forge($this->created_at)->format('H:i - Y/m/d'),
            Jalalian::forge($this->created_at)->ago(),
            Jalalian::forge($this->created_at)->getTimestamp(),
        ];
        return $date;
    }

    public function getJalaliUpdatedAtAttribute()
    {
        $date = [
            Jalalian::forge($this->updated_at)->format('%d %B %Y'),
            Jalalian::forge($this->updated_at)->format('Y/m/d'),
            Jalalian::forge($this->updated_at)->format('H:i - Y/m/d'),
            Jalalian::forge($this->updated_at)->ago(),
            Jalalian::forge($this->updated_at)->getTimestamp(),
        ];
        return $date;
    }

    public function getSafeAttribute()
    {
        $user = $this->user;

        return [
            'id' => $this->id,
            'answer' => $this->answer,
            'is_buyer' => $this->is_buyer,
            'vote_up' => $this->vote_up,
            'vote_down' => $this->vote_down,
            'status' => $this->status,
            'jalali_created_at' => $this->jalali_created_at,
            'jalali_updated_at' => $this->jalali_updated_at,

            'user' => [
                'id' => $user->id,
                'get_name' => $user->get_name,
                'username' => $user->username,
                'type' => $user->type,
                'seller' => $user->seller,
            ],
        ];
    }
    /** end append */

    /**
     * remove cache data in update
     */
    protected static function boot()
    {
        parent::boot();

        static::saved(function ($answer) {
            // product appends
            Cache::forget('product_answers_info_' . $answer->question->product_id);

            // question appends
            Cache::forget('question_answers_' . $answer->question_id);
        });

        static::deleted(function ($answer) {
            // product appends
            Cache::forget('product_answers_info_' . $answer->question->product_id);

            // question appends
            Cache::forget('question_answers_' . $answer->question_id);
        });
    }

    /* relationships **************/
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    public function question(): BelongsTo
    {
        return $this->belongsTo(Question::class);
    }
}
