<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Cache;
use Morilog\Jalali\Jalalian;

class Setting extends Model
{
    use HasFactory;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'key',
        'value',
    ];

    /**
     * additional data
     */
    protected $appends = [
        'get_image',
        'jalali_created_at',
        'jalali_updated_at',
    ];

    public function getGetImageAttribute()
    {
        $result = null;
        if ($this->value != null && is_array(@unserialize($this->value))) {
            $fileArray = unserialize($this->value);
            if (array_key_exists('id', $fileArray) && $fileArray['id'] > 0 && array_key_exists('url', $fileArray)) {
                $file = Cache::remember('files_' . $fileArray['id'], now()->addMonth(), function () use ($fileArray) {
                    return File::find([$fileArray['id']])->first();
                });
                if ($file != null) {
                    $result = asset($file->url);
                }
            }
        }
        return $result;
    }

    public function getJalaliCreatedAtAttribute()
    {
        $date = [
            Jalalian::forge($this->created_at)->format('%d %B %Y'),
            Jalalian::forge($this->created_at)->format('Y/m/d'),
            Jalalian::forge($this->created_at)->format('H:i - Y/m/d'),
            Jalalian::forge($this->created_at)->ago(),
            Jalalian::forge($this->created_at)->getTimestamp(),
        ];
        return $date;
    }

    public function getJalaliUpdatedAtAttribute()
    {
        $date = [
            Jalalian::forge($this->updated_at)->format('%d %B %Y'),
            Jalalian::forge($this->updated_at)->format('Y/m/d'),
            Jalalian::forge($this->updated_at)->format('H:i - Y/m/d'),
            Jalalian::forge($this->updated_at)->ago(),
            Jalalian::forge($this->updated_at)->getTimestamp(),
        ];
        return $date;
    }
    /** end append */

    /**
     * remove cache data in update
     */
    protected static function boot()
    {
        parent::boot();

        $settingsCaches = [
            'settings',
            ...array_map(function ($setting) {
                return 'setting_' . $setting['key'];
            }, Setting::select(['key'])->get()->makeHidden(['get_image', 'jalali_created_at', 'jalali_updated_at'])->toArray())
        ];

        static::saved(function () use ($settingsCaches) {
            // all settings
            foreach ($settingsCaches as $cache) {
                Cache::forget($cache);
            }
        });

        static::deleted(function () use ($settingsCaches) {
            // all settings
            foreach ($settingsCaches as $cache) {
                Cache::forget($cache);
            }
        });
    }
}
