<?php

namespace App\Models;

use Carbon\Carbon;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Support\Facades\Cache;
use Morilog\Jalali\Jalalian;

class Story extends Model
{
    use HasFactory;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'user_id',
        'type',
        'title',
        'cover',
        'file',
        'widgets',
        'like',
        'status',
        'expired_at',
    ];

    protected $casts = [
        'user_id' => 'integer',
        'like' => 'integer',
    ];

    /**
     * additional data
     */
    protected $appends = [
        'get_widgets',
        'is_liked',
        'is_expired',
        'jalali_expired_at',
        'jalali_created_at',
        'jalali_updated_at',
    ];

    public function getGetWidgetsAttribute()
    {
        $widgets = $this->widgets != null ? unserialize($this->widgets) : [
            'link' => [
                'title' => null,
                'url' => null,
            ],
            'product' => null
        ];

        if (array_key_exists('product', $widgets) && $widgets['product'] != null) {
            $product = Cache::remember('story_product_' . $widgets['product'], now()->addMonth(), function () use ($widgets) {
                return Product::where('status', 'publish')->where('uuid', $widgets['product'])->first();
            });
            if ($product != null) {
                $widgets['product'] = [
                    'id' => $product->id,
                    'uuid' => $product->uuid,
                    'title' => $product->title,
                    'slug' => $product->slug,
                    'get_images' => $product->get_images,
                    'comments_info' => $product->comments_info,
                    'in_stock_status' => $product->in_stock_status,
                    'inventory' => $product->inventory,
                    'best_normal_price' => $product->best_normal_price,
                    'unique_color' => $product->unique_color,
                    'best_price' => $product->best_price,
                    'has_discount' => $product->has_discount,
                    'is_vip' => $product->is_vip,
                ];
            } else {
                $widgets['product'] = null;
                $this->update([
                    'widgets' => serialize($widgets)
                ]);
            }
        }

        return $widgets;
    }

    public function getIsLikedAttribute()
    {
        $likedStories = session()->get('liked_stories', []);
        if (array_key_exists($this->id, $likedStories)) {
            return true;
        }
        return false;
    }

    public function getIsExpiredAttribute()
    {
        return $this->expired_at >= Carbon::now()->toDateString() ? false : true;
    }

    public function getJalaliExpiredAtAttribute()
    {
        $date = [
            Jalalian::forge($this->expired_at)->format('%d %B %Y'),
            Jalalian::forge($this->expired_at)->format('Y/m/d'),
            Jalalian::forge($this->expired_at)->format('H:i - Y/m/d'),
            Jalalian::forge($this->expired_at)->ago(),
            Jalalian::forge($this->expired_at)->getTimestamp(),
        ];
        return $date;
    }

    public function getJalaliCreatedAtAttribute()
    {
        $date = [
            Jalalian::forge($this->created_at)->format('%d %B %Y'),
            Jalalian::forge($this->created_at)->format('Y/m/d'),
            Jalalian::forge($this->created_at)->format('H:i - Y/m/d'),
            Jalalian::forge($this->created_at)->ago(),
            Jalalian::forge($this->created_at)->getTimestamp(),
        ];
        return $date;
    }

    public function getJalaliUpdatedAtAttribute()
    {
        $date = [
            Jalalian::forge($this->updated_at)->format('%d %B %Y'),
            Jalalian::forge($this->updated_at)->format('Y/m/d'),
            Jalalian::forge($this->updated_at)->format('H:i - Y/m/d'),
            Jalalian::forge($this->updated_at)->ago(),
            Jalalian::forge($this->updated_at)->getTimestamp(),
        ];
        return $date;
    }
    /** end append */

    /* relationships **************/
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }
}
