<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Morilog\Jalali\Jalalian;

class Ticket extends Model
{
    use HasFactory;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'user_id',

        'title',
        'priority',
        'status',
    ];

    protected $casts = [
        'user_id' => 'integer',
    ];

    /**
     * additional data
     */
    protected $appends = [
        'ticket_answers',
        'jalali_created_at',
        'jalali_updated_at',

        'safe',
    ];

    public function getTicketAnswersAttribute()
    {
        return $this->ticketAnswers()->get()->map(fn($answer) => [
            'id' => $answer->id,
            'user_id' => $answer->user_id,
            'ticket_id' => $answer->ticket_id,
            'type' => $answer->type,
            'content' => $answer->content,
            'get_file' => $answer->get_file,
            'jalali_created_at' => $answer->jalali_created_at,
            'jalali_updated_at' => $answer->jalali_updated_at,
            'status' => $answer->status,
            'user' => $answer->user->safe,
        ]);
    }

    public function getJalaliCreatedAtAttribute()
    {
        $date = [
            Jalalian::forge($this->created_at)->format('%d %B %Y'),
            Jalalian::forge($this->created_at)->format('Y/m/d'),
            Jalalian::forge($this->created_at)->format('H:i - Y/m/d'),
            Jalalian::forge($this->created_at)->ago(),
            Jalalian::forge($this->created_at)->getTimestamp(),
        ];
        return $date;
    }

    public function getJalaliUpdatedAtAttribute()
    {
        $date = [
            Jalalian::forge($this->updated_at)->format('%d %B %Y'),
            Jalalian::forge($this->updated_at)->format('Y/m/d'),
            Jalalian::forge($this->updated_at)->format('H:i - Y/m/d'),
            Jalalian::forge($this->updated_at)->ago(),
            Jalalian::forge($this->updated_at)->getTimestamp(),
        ];
        return $date;
    }

    public function getSafeAttribute()
    {
        return [
            'id' => $this->id,
            'user_id' => $this->user_id,
            'title' => $this->title,
            'priority' => $this->priority,
            'status' => $this->status,
            'ticket_answers' => $this->ticket_answers,
            'jalali_created_at' => $this->jalali_created_at,
            'jalali_updated_at' => $this->jalali_updated_at,
            'user' => $this->user->safe,
        ];
    }
    /** end append */

    /* relationships **************/
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    public function ticketAnswers(): HasMany
    {
        return $this->hasMany(TicketAnswer::class);
    }
}
