<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Support\Facades\Cache;
use Morilog\Jalali\Jalalian;

class TicketAnswer extends Model
{
    use HasFactory;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'user_id',
        'ticket_id',

        'type',
        'content',
        'file',
        'status',
    ];

    protected $casts = [
        'user_id' => 'integer',
        'ticket_id' => 'integer',
    ];

    /**
     * additional data
     */
    protected $appends = [
        'get_file',
        'jalali_created_at',
        'jalali_updated_at',

        'safe',
    ];

    public function getGetFileAttribute()
    {
        $result = null;
        if ($this->type != null && $this->file != null) {
            $file = Cache::remember('files_' . $this->file, now()->addMonth(), function () {
                return File::find($this->file);
            });

            if ($file != null) {
                $result = [
                    'link' => asset($file->url),
                    'alt' => $file->description,
                    'name' => $file->name,
                    'size' => ($file->size / 1000) . 'mb',
                ];
            } else if ($this->type == "file") {
                $result = [
                    'link' => asset('img/temp/corrupt_file.png'),
                    'alt' => __('messages.word.deleted_image'),
                    'name' => __('messages.word.deleted_image'),
                    'size' => __('messages.word.deleted'),
                ];
            }
        }

        return $result;
    }

    public function getJalaliCreatedAtAttribute()
    {
        $date = [
            Jalalian::forge($this->created_at)->format('%d %B %Y'),
            Jalalian::forge($this->created_at)->format('Y/m/d'),
            Jalalian::forge($this->created_at)->format('H:i - Y/m/d'),
            Jalalian::forge($this->created_at)->ago(),
            Jalalian::forge($this->created_at)->getTimestamp(),
        ];
        return $date;
    }

    public function getJalaliUpdatedAtAttribute()
    {
        $date = [
            Jalalian::forge($this->updated_at)->format('%d %B %Y'),
            Jalalian::forge($this->updated_at)->format('Y/m/d'),
            Jalalian::forge($this->updated_at)->format('H:i - Y/m/d'),
            Jalalian::forge($this->updated_at)->ago(),
            Jalalian::forge($this->updated_at)->getTimestamp(),
        ];
        return $date;
    }

    public function getSafeAttribute()
    {
        return [
            'id' => $this->id,
            'user_id' => $this->user_id,
            'ticket_id' => $this->ticket_id,
            'type' => $this->type,
            'content' => $this->content,
            'get_file' => $this->get_file,
            'jalali_created_at' => $this->jalali_created_at,
            'jalali_updated_at' => $this->jalali_updated_at,
            'status' => $this->status,
            'user' => $this->user->safe,
        ];
    }
    /** end append */

    /* relationships **************/
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    public function ticket(): BelongsTo
    {
        return $this->belongsTo(Ticket::class);
    }
}
