<?php

namespace App\Payments\Drivers;

use GuzzleHttp\Client;
use GuzzleHttp\RequestOptions;
use Illuminate\Support\Facades\Log;
use Shetabit\Multipay\Abstracts\Driver;
use Shetabit\Multipay\Contracts\ReceiptInterface;
use Shetabit\Multipay\Exceptions\InvalidPaymentException;
use Shetabit\Multipay\Exceptions\PurchaseFailedException;
use Shetabit\Multipay\Invoice;
use Shetabit\Multipay\Receipt;
use Shetabit\Multipay\RedirectionForm;
use Shetabit\Multipay\Request;

/**
 * Kipaa Custom Driver
 *
 * Flow:
 *   purchase() -> pay() -> callback (payment_token, reciept_number, state, msg) -> verify()
 */
class Kipaa extends Driver
{
    /** Base relative endpoints (append to baseUrl in settings) */
    private const REQUEST_TOKEN_ENDPOINT  = 'request_payment_token';
    private const VERIFY_ENDPOINT         = 'verify_transaction';
    private const CONFIRM_ENDPOINT        = 'confirm_transaction';
    private const INQUIRE_ENDPOINT        = 'get_transaction';

    /** @var Client */
    protected Client $client;

    /** @var Invoice */
    protected $invoice;

    /** @var object Driver settings */
    protected $settings;

    /** @var string|null Payment URL returned by request_payment_token */
    protected $paymentUrl = null;

    /**
     * Constructor.
     *
     * @param Invoice $invoice
     * @param array   $settings
     */
    public function __construct(Invoice $invoice, $settings)
    {
        $this->invoice($invoice);              // مطابق الگوی Digipay (ست کردن invoice داخلی Driver)
        $this->settings = (object) $settings;

        $base = rtrim($this->settings->baseUrl ?? '', '/') . '/';

        $this->client = new Client([
            'base_uri'    => $base,
            'timeout'     => $this->settings->timeout ?? 30,
            'http_errors' => false,
        ]);
    }

    /**
     * Purchase (Request payment token).
     *
     * Must return the transaction id (payment_token).
     *
     * @return string
     * @throws PurchaseFailedException
     */
    public function purchase(): string
    {
        // Amount from invoice (assumed stored in Toman if currency == 'T')
        $amount = (int) $this->invoice->getAmount();
        if (($this->settings->currency ?? 'T') === 'T') {
            $amount *= 10; // Convert to Rial
        }

        $merchantOrderId = $this->invoice->getUuid()
            ?? $this->invoice->getTransactionId()
            ?? uniqid('kipaa_', true);

        $mobile  = $this->invoice->getDetail('mobile')
            ?? $this->invoice->getDetail('cellphone')
            ?? $this->invoice->getDetail('phone');
        $details = $this->invoice->getDetail('details');

        $payload = [
            'amount'        => $amount,
            'callback_url'  => $this->settings->callbackUrl,
            'merchant_order_id' => $merchantOrderId,
        ];

        if ($mobile) {
            $payload['mobile'] = $mobile;
        }
        if ($details) {
            $payload['details'] = is_string($details)
                ? $details
                : json_encode($details, JSON_UNESCAPED_UNICODE);
        }

        try {
            $response = $this->client->request('POST', self::REQUEST_TOKEN_ENDPOINT, [
                RequestOptions::FORM_PARAMS => $payload,
                RequestOptions::HEADERS     => $this->headers(),
            ]);

            $status = $response->getStatusCode();
            $body   = json_decode($response->getBody()->getContents(), true);

            if (
                $status !== 200 ||
                empty($body['Success']) ||
                empty($body['Content']['payment_token']) ||
                empty($body['Content']['payment_url'])
            ) {
                $message = $body['Message'] ?? "Kipaa purchase failed (HTTP $status).";
                throw new PurchaseFailedException($message);
            }

            $paymentToken     = $body['Content']['payment_token'];
            $this->paymentUrl = $body['Content']['payment_url'];

            // set transaction id on invoice (framework will run user callback with this id)
            $this->invoice->transactionId($paymentToken);

            return $paymentToken;
        } catch (\Throwable $e) {
            throw new PurchaseFailedException($e->getMessage(), (int)$e->getCode(), $e);
        }
    }

    /**
     * Pay: returns a RedirectionForm.
     *
     * If Kipaa requires POST with payment_token, we send it; otherwise switch method to GET and empty inputs.
     *
     * @return RedirectionForm
     */
    public function pay(): RedirectionForm
    {
        if (!$this->paymentUrl) {
            throw new \RuntimeException('Kipaa: paymentUrl is not set. Call purchase() first.');
        }

        return $this->redirectWithForm(
            $this->paymentUrl,
            ['payment_token' => $this->invoice->getTransactionId()],
            'POST'
        );
    }

    /**
     * Verify (and confirm) the transaction.
     *
     * @return ReceiptInterface
     * @throws InvalidPaymentException
     */
    public function verify(): ReceiptInterface
    {
        $paymentToken  = Request::input('payment_token', $this->invoice->getTransactionId());
        $receiptNumber = Request::input('reciept_number'); // املای API: reciept_number
        $state         = (int) Request::input('state');
        $stateMsg      = Request::input('msg');

        if (!$paymentToken || !$receiptNumber) {
            throw new InvalidPaymentException('Missing payment_token or reciept_number in callback.');
        }

        if ($state !== 100) {
            throw new InvalidPaymentException("Kipaa state not successful: {$state} ({$stateMsg})");
        }

        // Step 1: verify_transaction
        $verifyData = $this->call(self::VERIFY_ENDPOINT, [
            'payment_token'  => $paymentToken,
            'reciept_number' => $receiptNumber,
        ]);

        if (empty($verifyData['success'])) {
            $msg = $verifyData['message'] ?? 'Verification failed.';
            throw new InvalidPaymentException($msg);
        }

        $amount        = $verifyData['content']['Amount']          ?? null;
        $creditAmount  = $verifyData['content']['CreditAmount']    ?? null;
        $cashAmount    = $verifyData['content']['CashAmount']      ?? null;
        $merchantOrder = $verifyData['content']['MerchantOrderID'] ?? null;

        // Step 2: confirm_transaction (finalize)
        $confirmData = $this->call(self::CONFIRM_ENDPOINT, [
            'payment_token'  => $paymentToken,
            'reciept_number' => $receiptNumber,
        ]);

        if (empty($confirmData['success'])) {
            $msg = $confirmData['message'] ?? 'Confirm transaction failed.';
            throw new InvalidPaymentException($msg);
        }

        $confirmNumber = $confirmData['content']['ConfirmTransactionNumber'] ?? null;

        $receipt = new Receipt('kipaa', $paymentToken);
        $receipt->detail([
            'reciept_number'         => $receiptNumber,
            'confirm_transaction_no' => $confirmNumber,
            'amount_rial'            => $amount,
            'credit_amount_rial'     => $creditAmount,
            'cash_amount_rial'       => $cashAmount,
            'merchant_order_id'      => $merchantOrder,
            'state'                  => $state,
            'state_message'          => $stateMsg,
        ]);

        return $receipt;
    }

    /**
     * Optional: public inquire method if you need later status checking.
     */
    public function inquire(string $paymentToken, string $receiptNumber): array
    {
        return $this->call(self::INQUIRE_ENDPOINT, [
            'payment_token'  => $paymentToken,
            'reciept_number' => $receiptNumber,
        ]);
    }

    /* -----------------------------------------------------------
     | Internal helpers
     | ----------------------------------------------------------- */

    /**
     * Generic API POST call.
     *
     * @param string $endpoint Relative endpoint (already baseUri in client)
     * @param array  $params   Form params
     * @return array
     * @throws InvalidPaymentException
     */
    private function call(string $endpoint, array $params): array
    {
        try {
            $response = $this->client->request('POST', $endpoint, [
                RequestOptions::FORM_PARAMS => $params,
                RequestOptions::HEADERS     => $this->headers(),
            ]);

            $status = $response->getStatusCode();
            $body   = json_decode($response->getBody()->getContents(), true);

            if ($status !== 200) {
                $msg = $body['message'] ?? "Kipaa API error ($endpoint) HTTP $status.";
                throw new InvalidPaymentException($msg, $status);
            }

            return $body ?: [];
        } catch (\Throwable $e) {
            throw new InvalidPaymentException("Kipaa API call error ($endpoint): " . $e->getMessage(), 0, $e);
        }
    }

    /**
     * Default headers for all API calls.
     *
     * @return array
     */
    private function headers(): array
    {
        return [
            'Accept'        => 'application/json',
            'Authorization' => 'Bearer ' . ($this->settings->apiKey ?? ''),
        ];
    }
}
